import logging
import asyncio

from geoffrey import plugin
from geoffrey.data import State, EventType
from geoffrey.utils import execute
from geoffrey.subscription import subscription


class SnakeFood(plugin.GeoffreyPlugin):
    """
    snakefood plugin.

    """
    @subscription
    def modified_files(self, event):
        """
        Subscription criteria.

        Should be used as annotation of plugin.Tasks arguments.
        Can be used multiple times.

        """
        return (self.project.name == event.project and
                event.plugin == "filecontent" and
                event.key.endswith('.py') and
                event.type in (EventType.created, EventType.modified))

    @subscription
    def sfood_data(self, event):
        """
        New or modified events generated by `run_sfood`.

        """
        return (self.project.name == event.project and
                event.task == "sfood" and
                event.type in (EventType.created, EventType.modified))

    @asyncio.coroutine
    def run_sfood(self, events:"modified_files") -> plugin.Task:
        """
        Run sfood when python files are modified.

        """
        sfood_path = self.config.get(self._section_name, "sfood_path")

        while True:
            event = yield from events.get()

            filename = event.key

            exitcode, stdout, stderr = yield from execute(
                sfood_path, '-i', '-q', filename)

            state = self.new_state(task='sfood', key=filename,
                          dependencies=stdout.decode('utf-8'))

            yield from self.hub.put(state)

    @asyncio.coroutine
    def run_sfoodgraph(self, events:"sfood_data") -> plugin.Task:
        """
        Run sfood-graph when sfood's output is changed

        """
        sfoodgraph_path = self.config.get(self._section_name,
                                          "sfoodgraph_path")
        while True:
            event = yield from events.get()

            filename = event.key

            exitcode, stdout, stderr = yield from execute(
                sfoodgraph_path, '-p',
                stdin=event.dependencies.encode('utf-8'))

            state = self.new_state(task='sfoodgraph', key=filename,
                                   graph=stdout.decode('utf-8'))

            yield from self.hub.put(state)
