#!/usr/bin/env python
# encoding: utf-8
'''
bin.gus_create_feature_file -- Creates a feature File using acceptance criteria on user stories in Gus

bin.gus_create_feature_file is a Utility to extract When/Then formatted Acceptance criteria in user stories and
    output a format that can then be used as a feature file that is properly annotated allowing for updating of
    test execution metadata when it is run

@author:     Shawn

@copyright:  2014 Salesforce.com. All rights reserved.

@license:    license

@contact:    scrosby@salesforce.com
@deffield    updated: Sept 13/2014
'''

import sys
import os, re, traceback

from optparse import OptionParser
from gus.BacklogClient import BacklogClient
from gherkin.GherkinGusAdapter import GherkinGusAdapter

__all__ = []
__version__ = 0.1
__date__ = '2014-08-22'
__updated__ = '2014-09-13'

gus = BacklogClient()

def main(argv=None):
    '''Command line options.'''

    program_name = os.path.basename(sys.argv[0])
    program_version = "v0.1"
    program_build_date = "%s" % __updated__

    program_version_string = '%%prog %s (%s)' % (program_version, program_build_date)
    #program_usage = '''usage: spam two eggs''' # optional - will be autogenerated by optparse
    program_longdesc = '''''' # optional - give further explanation about what the program does
    program_license = "Copyright 2014 Shawn (scrosby@salesforce.com)                                            \
                Licensed under the Apache License 2.0\nhttp://www.apache.org/licenses/LICENSE-2.0"

    if argv is None:
        argv = sys.argv[1:]
    try:
        # setup option parser
        parser = OptionParser(version=program_version_string, epilog=program_longdesc, description=program_license)
        parser.add_option("-t", "--team", dest="team", help="ID of team to generate feature files for entire sprint")
        parser.add_option("-o", "--output", dest="output", help="Directory to output feature files to otherwise STDOUT")

        # process options
        (opts, args) = parser.parse_args(argv)
        
        if opts.output:
            if not os.path.isdir(opts.output):
                raise Exception("Output needs to be a valid directory name")

        # MAIN BODY #
        
        if opts.team is not None:
            sprint = gus.get_current_sprint_for_team(opts.team)
            work_list = gus.get_user_stories_for_sprint(sprint)
        else:
            work_list = []    
            for arg in args:
                work = gus.find_work(arg)
                work_list.append(work['Id'])
            
        for work in work_list:
            w_record = gus.get_work_record(work)
            feature = w_record['Subject__c']
            from_gus = GherkinGusAdapter(feature, w_record['Name'], w_record['Details__c'])
            
            all_ac = gus.get_acceptance_criteria_for_work(work)
            for ac in all_ac:
                acceptance = gus.get_acceptance_criterion_record(ac)
                from_gus.add_scenario(acceptance['Name'], acceptance['Description__c'])

            # look for parent scenarios
            recurse_parent_scenarios(work, from_gus)
                    

            if opts.output is not None:
                filename = re.sub("[\ \/&<>]", "_", feature).lower() + ".feature"
                with open(os.path.join(opts.output, filename), 'wb') as output:
                    output.write(from_gus.to_object().to_string())
                    output.flush()
                    output.close()
            else:
                print(from_gus.to_object().to_string())

    except Exception as e:
        traceback.print_exc()
        indent = len(program_name) * " "
        sys.stderr.write(program_name + ": " + repr(e) + "\n")
        sys.stderr.write(indent + "  for help use --help")
        return 2

def recurse_parent_scenarios(work, gherkin_adapter):
    parent_ids = gus.get_parent_work_for_work(work)
    for p_id in parent_ids:
        recurse_parent_scenarios(p_id, gherkin_adapter)
        w_rec = gus.get_work_record(p_id)
        p_ac = gus.get_acceptance_criteria_for_work(p_id)
        for ac in p_ac:
            s_rec = gus.get_acceptance_criterion_record(ac)
            gherkin_adapter.add_scenario(s_rec['Name'], s_rec['Description__c'], w_rec['Name'])

if __name__ == "__main__":
    sys.exit(main())