#!/usr/bin/env python
# encoding: utf-8
'''
bin.gherkin_sync_user_story -- Synchronizes a feature file with GUS

bin.gherkin_sync_user_story is a utility that reads a local feature file and uses an annotated
    GUS id to log into gus and compare with acceptance criteria on a user story.  It will update
    GUS with local changes and create new acceptance criteria if a scenario doesn't exist in GUS.

It defines classes_and_methods

@author:     Shawn

@copyright:  2014 Salesforce.com. All rights reserved.

@license:    license

@contact:    scrosby@salesforce.com
@deffield    updated: Sept 13/2014
'''

import sys
import os, traceback

from optparse import OptionParser
from gherkin.GherkinAdapter import GherkinAdapter, ScenarioAdapter
from gherkin.Gherkin import Scenario
from gus.BacklogClient import BacklogClient

__all__ = []
__version__ = 0.1
__date__ = '2014-09-01'
__updated__ = '2014-09-13'

def main(argv=None):
    '''Command line options.'''

    program_name = os.path.basename(sys.argv[0])
    program_version = "v0.1"
    program_build_date = "%s" % __updated__

    program_version_string = '%%prog %s (%s)' % (program_version, program_build_date)
    #program_usage = '''usage: spam two eggs''' # optional - will be autogenerated by optparse
    program_longdesc = '''''' # optional - give further explanation about what the program does
    program_license = "Copyright 2014 Shawn (Salesforce.com)                                            \
                Licensed under the Apache License 2.0\nhttp://www.apache.org/licenses/LICENSE-2.0"

    if argv is None:
        argv = sys.argv[1:]
    try:
        # setup option parser
        parser = OptionParser(version=program_version_string, epilog=program_longdesc, description=program_license)

        # process options
        (opts, args) = parser.parse_args(argv)
        gus = BacklogClient()
        
        # MAIN BODY #
        for arg in args:
            with open(arg, "rb") as feature_file:
                data = feature_file.read()
                feature_file.close()
                
            gherkin = GherkinAdapter(data).to_object()
            work = gus.find_work(gherkin.gus_id)
            
            # get the acceptance criteria for the story
            all_ac = gus.get_acceptance_criteria_for_work(work['Id'])
            ac_data = {}
            for ac in all_ac:
                ac_record = gus.get_acceptance_criterion_record(ac)
                ac_data[ac_record['Name']] = ac_record
            
            scenarios = gherkin.scenarios
            
            if gherkin.background_steps is not None:
                scenarios.append(gherkin.background_steps)
            
            # First see if there is some local work to get uploaded
            for scenario in scenarios:
                try:
                    name = scenario.name
                    description = "\n".join([x.to_string() for x in scenario.steps])
                    if scenario.is_outline:
                        description += "\n\n{}".format(scenario.example_string())
                    
                    interested_parties = []
                    
                    # the power of three
                    for party in [work['Assignee__c'], work['QA_Engineer__c'],work['Product_Owner__c']]:
                        if party is not None and party not in interested_parties and party != gus.get_current_user_id():
                            interested_parties.append(party)
                    
                    if name not in ac_data.keys() and scenario.for_gus_id is None:
                        print("Acceptance Criteria [{}] Not in Gus, Adding it.".format(name))
                        # create acceptance criteria
                        ac_id = gus.add_acceptance_criteria(work['Id'], name, description)
                        gus.post_chatter_message("New Acceptance Criteria Added Because of new Test Code: {}".format(name), ac_id, interested_parties)
                    else:
                        if scenario.for_gus_id:
                            # if this is inherited find the AC in GUS
                            s_work = gus.find_work(scenario.for_gus_id)
                            for party in [s_work['Assignee__c'], s_work['QA_Engineer__c'],s_work['Product_Owner__c']]:
                                if party is not None and party not in interested_parties and party != gus.get_current_user_id():
                                    interested_parties.append(party)
                            ac_id = gus.get_specific_acceptance_criteria(s_work['Id'], scenario.name)
                            ac_record = gus.get_acceptance_criterion_record(ac_id)
                        else:
                            ac_record = ac_data[name]
                            
                        if description.replace("\r\n", "\n") != ac_record['Description__c'].replace("\r\n","\n"):
                            print("Acceptance Criteria [{}] Doesn't match, updating GUS".format(ac_record['Name']))
                            gus.update_acceptance_criteria_details(ac_record['Id'], description)
                            gus.post_chatter_message("Acceptance Criteria Updated by Change to Test Code\n\n{}".format(description), ac_record['Id'], interested_parties)
                except Exception as e:
                    traceback.print_exc()
                    print(scenario.to_string())
            
            # Then check to see if we are missing some new Acceptance Criteria in GUS
            scenario_names = [x.name for x in scenarios]
            for ac_name in ac_data.keys():
                if ac_name not in scenario_names:
                    print("Acceptance Criteria [{}] Not In Feature File, Appending".format(ac_name))
                    scenario_data = ScenarioAdapter(ac_data[ac_name]['Description__c'])
                    scenario = Scenario()
                    scenario.name = ac_name
                    for step in scenario_data.get_steps():
                        scenario.steps.append(step.to_object())
                    examples = scenario_data.get_examples()
                    if examples is not None:
                        scenario.is_outline = True
                        scenario.examples.extend(examples)
                        
                    with open(arg, "a+b") as feature_file:
                        feature_file.write("\n{}\n".format(scenario.to_string()))
                        feature_file.flush()
                        feature_file.close()
                    

    except Exception, e:
        traceback.print_exc()
        indent = len(program_name) * " "
        sys.stderr.write(program_name + ": " + repr(e) + "\n")
        sys.stderr.write(indent + "  for help use --help")
        return 2


if __name__ == "__main__":
    sys.exit(main())