'''
Created on Sep 1, 2014

@author: scrosby
'''
import re
from gherkin.Gherkin import Gherkin, Scenario, Step

class GherkinAdapter:
    '''
    Reads a feature file and parses out scenarios and steps to create a Gherkin object
    '''
    def __init__(self, feature_file_data):
        '''
        @param feature_file_data: The text of a feature file
        '''
        self.data = feature_file_data.replace("\r\n", "\n")
        
    def get_gus_id(self):
        '''
        Parses the GUS ID that is annotated on the file as @gus:W-xxxxxxx
        '''
        match = re.search("@gus:(W-\d{7})", self.data)
        return match.group(1)
    
    def __find_scenario_annotations__(self, scenario):
        out = None
        match = re.search("((@.+?)\n)+?{}".format(scenario), self.data, flags=re.MULTILINE)
        if match:
            out = match.group(2)
            
        return out
        
    def get_scenarios(self):
        '''
        Does a greedy search for the all the scenarios in a feature file and then
        assumes that the next line it encounters that begins with the word Scenario
        is a new scenario
        '''
        out = []
        match = re.search("(Scenario( Outline)?:.*)$", self.data, flags=re.DOTALL|re.M)
        scenarios = match.group(1)
        lines = scenarios.split("\n")
        annotations = None
        scenario = []
        for line in lines:
            if re.search("^\s*Scenario", line) is not None:
                if len(scenario) > 0:
                    if scenario[-1] == "":
                        scenario.pop()
                    out.append(ScenarioAdapter("\n".join(scenario), annotations))
                    annotations = None
                    scenario = []
                annotations = self.__find_scenario_annotations__(line)
            scenario.append(line)
        out.append(ScenarioAdapter("\n".join(scenario), annotations))
        
        return out
    
    def get_feature_name(self):
        '''
        Returns the name of the feature
        '''
        match = re.search("Feature: (.*)\n", self.data)
        return match.group(1)
    
    def get_backgroud_steps(self):
        '''
        Returns the Background steps to be performed for each scenario
        '''
        match = re.search("Background:(.*?)\n\n", self.data, flags=re.DOTALL)
        if match is not None:
            return ScenarioAdapter(match.group(1))
        else:
            return None
        
    def to_object(self):
        '''
        Constructs a Gherkin object by parsing the entire file
        '''
        gherkin = Gherkin()
        gherkin.feature_name = self.get_feature_name()
        gherkin.gus_id = self.get_gus_id()
        if self.get_backgroud_steps() is not None:
            gherkin.background_steps = Scenario()
            gherkin.background_steps.name = "Background"
            for step in self.get_backgroud_steps().get_steps():
                gherkin.background_steps.steps.append(step.to_object())
        for scenario in self.get_scenarios():
            gherkin.scenarios.append(scenario.to_object())
            
        return gherkin
    
class ScenarioAdapter:
    '''
    Parses the text of a scenario
    '''
    def __init__(self, scenario_text, annotations=None):
        '''
        @param scenario_text: the steps of a scenario with or without heading
        '''
        self.data = scenario_text.replace("\r\n", "\n")
        self.annotation_data = annotations
        
    def get_name(self):
        '''
        Returns the name of the scenario.  If it cant find one, it assumes it is the background
        '''
        match = re.search("Scenario( Outline)?: (.*)\n", self.data)
        if match is not None:
            return match.group(2)
        else:
            return "Background"
    
    def is_outline(self):
        '''
        Returns True or False depending on the heading
        '''
        match = re.search("Scenario Outline:", self.data)
        return match is not None
    
    def get_examples(self):
        '''
        Returns an array of dictionaries in order to construct a table
        used in a scenario outline
        '''
        out = None
        match = re.search("Examples:\s*\n(.*)", self.data, flags=re.DOTALL|re.MULTILINE)
        if match:
            out = []
            lines = match.group(1).strip().split("\n")
            headings = [ x.strip() for x in lines[0].strip(" |").strip().split("|")]
            for line in lines[1:]:
                detail = {}
                values = [ x.strip() for x in line.strip(" |").strip().split("|")]
                x = 0
                for h in headings:
                    detail[h] = values[x]
                    x+=1
                out.append(detail)
        return out
    
    def get_steps(self):
        '''
        Returns all the steps in a scenario (beginning with Given,When,Then,But,And)
        '''
        out = []
        matches = re.finditer("^\s*((Given|When|Then|And|But) .*)", self.data, flags=re.MULTILINE)
        for match in matches:
            out.append(StepAdapter(match.group(1)))
        return out
    
    def get_description(self):
        '''
        Re-assembles the steps and examples into a scenario.
        '''
        out = "\n".join([x.get_description() for x in self.get_steps()])
        if self.is_outline():
            out += "\n\nExamples:\n"
            examples = self.get_examples()
            out += "|"
            for h in examples[0].keys():
                out += " {} |".format(h)
                
            for example in examples:
                out += "\n|"
                for v in example.values():
                    out += " {} |".format(v)
        return out

    def get_gus_id(self):
        out = None
        if self.annotation_data:
            match = re.search("@gus_inherited:(W-\d{7})", self.annotation_data)
            if match is not None:
                out = match.group(1)
                
        return out
        
    def to_object(self):
        '''
        Parses the text into a Scenario object
        '''
        scenario = Scenario()
        scenario.name = self.get_name()
        scenario.is_outline = self.is_outline()
        scenario.for_gus_id = self.get_gus_id()
        for step in self.get_steps():
            scenario.steps.append(step.to_object())
        
        if self.get_examples() is not None:   
            scenario.examples.extend(self.get_examples()) 
        return scenario

class StepAdapter:
    '''
    Parses a step into its parts
    '''
    def __init__(self, line):
        '''
        @param line: The text of a step
        '''
        self.data = line
        match = re.search("^(Given|When|Then|And|But) (.*)", self.data)
        self.prefix = match.group(1)
        self.text = match.group(2)
        
    def get_description(self):
        '''
        Re-assembles a step object into a string
        '''
        return "{} {}".format(self.prefix, self.text)
        
    def to_object(self):
        '''
        Parses the string into a Step object
        '''
        step = Step()
        step.prefix = self.prefix
        step.text = self.text
        return step
    