#
# ginvoke: a programmable launcher for GTK+
# Copyright (C) 2011 Thomas Lee
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import sys
import os
import signal
import atexit

import gtk
import gobject

from . import NAME, VERSION
from .ui import Window, PopupMenu, StatusIcon
from .engine import Engine
from .settings import Settings
from .network import Server, Client

def toggle_visible(widget):
    if widget.get_visible():
        widget.hide()
    else:
        widget.present()

def toggle_visible_handler(widget):
    def impl(sender, *args):
        toggle_visible(widget)
    return impl

server_connection = toggle_visible_handler
window_deleteevent = toggle_visible_handler
statusicon_activate = toggle_visible_handler

def window_focusoutevent(window):
    def focusoutevent(sender, *args):
        if window.get_visible():
            window.hide()
    return focusoutevent

def statusicon_popupmenu(menu):
    def impl(sender, button, timestamp):
        menu.popup(None, None, None, button, timestamp)
    return impl

def makesighandler(server):
    def sighandler(*args):
        server.stop()
        sys.exit(0)
    return sighandler

def signals(server):
    sighandler = makesighandler(server)
    signal.signal(signal.SIGQUIT, sighandler)
    signal.signal(signal.SIGTERM, sighandler)
    signal.signal(signal.SIGINT, sighandler)
    atexit.register(sighandler)

def run():
    settings = Settings()
    settings.load()
    title = "%s v%d.%d.%d" % ((NAME,) + VERSION)

    server = Server(settings)
    if server.running:
        signals(server)
    else:
        # notify the server that we tried to start up, so that it
        # knows the user wants to see the UI
        client = Client(settings)
        connected = client.connected
        client.close()
        sys.exit(0)

    engine = Engine(server, settings)
    
    window = Window(engine)
    server.connect('connection', server_connection(window))
    window.set_title(title)
    window.connect("delete-event", window_deleteevent(window))

    popupmenu = PopupMenu()
    popupmenu.onquit(gtk.main_quit)

    statusicon = StatusIcon(title)
    statusicon.connect("activate", statusicon_activate(window))
    statusicon.connect("popup-menu", statusicon_popupmenu(popupmenu))

    if engine.settings.show_at_startup:
        window.show_all()
        window.present()
    statusicon.set_visible(True)

    try:
        gtk.main()
    except KeyboardInterrupt:
        pass

