#
# ginvoke: a programmable launcher for GTK+
# Copyright (C) 2011 Thomas Lee
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import os
import re
import logging

import gobject

from urllib import quote_plus as urlquote


class Engine(gobject.GObject):
    VAR_REGEX = re.compile(r"\${([^}]+)}")

    def __init__(self, server, settings):
        gobject.GObject.__init__(self)

        self.server = server
        self.settings = settings
        self.commandparser = CommandParser()

    def execute(self, window, command):
        rawargs, args = self.commandparser.parse(os.environ, command)
        cmdname, args = args[0], args[1:]
        context = ExecutionContext(
            command=cmdname,
            env=dict(os.environ),
            rawargs=rawargs,
            args=args,
            u=urlquote,
            engine=self,
            settings=self.settings,
            server=self.server,
            window=window
        )
        handler, params = self.settings.commands[cmdname]
        return handler(context, self._exprsubst(context, params))

    def _expr(self, context, arg):
        def subn_cb(match):
            try:
                return str(eval(match.group(1), {}, context))
            except KeyError:
                return match.group(0)
        return self.VAR_REGEX.subn(subn_cb, arg)[0]

    def _exprsubst(self, context, args):
        return tuple([self._expr(context, arg) for arg in args])

class ExecutionContext(object):
    def __init__(self, command, env, rawargs, args, u, engine, settings, server, window):
        object.__init__(self)
        self.env = env
        self.rawargs = rawargs
        self.args = args
        self.u = u
        self.engine = engine
        self.settings = settings
        self.server = server
        self.command = command
        self.window = window

    def __getitem__(self, key):
        try:
            return getattr(self, key)
        except KeyError:
            raise AttributeError("no such attribute: %s" % key)

class BadCommandException(Exception):
    pass

class EndOfCommandException(Exception):
    pass

class UnexpectedEndOfCommandException(Exception):
    pass

class CommandParser(object):
    ENV_REGEX = re.compile(r"\$([a-zA-Z_][a-zA-Z0-9_]*)")
    ENV2_REGEX = re.compile(r"\${([a-zA-Z_][a-zA-Z0-9_]*)}")

    def __init__(self):
        object.__init__(self)

        self.command = None
        self.offset = None

    def parse(self, env, command):
        self.command = command
        self.offset = 0
        rawargs = [arg for arg in self]
        return rawargs, [self.envsubst(env, arg) for arg in rawargs]

    def __iter__(self):
        try:
            while True:
                yield self.getnext()
        except EndOfCommandException:
            pass

    def envsubst(self, env, arg):
        def subn_cb(match):
            try:
                return env[match.group(1)]
            except KeyError:
                return match.group(0)
        return self.ENV_REGEX.subn(subn_cb, self.ENV2_REGEX.subn(subn_cb, arg)[0])[0]

    def getnext(self):
        c = self.c
        n = self.n

        self.checkeoc()

        while c().isspace():
            n()

        if c() in ('"', "'"):
            return self.quoted()
        else:
            return self.simple()

    def parsewhile(self, pred, eoc, escape, skiplast):
        c = self.c
        n = self.n

        escaped = False
        temp = ""
        try:
            while pred(c(), escaped):
                if escaped:
                    if escape: temp += escape(c())
                    escaped = False
                else:
                    if c() == "\\":
                        escaped = True
                    else:
                        temp += c()
                n()
            if skiplast and not self.iseoc(): n()
        except EndOfCommandException:
            if eoc:
                eoc(temp, escaped)
            else:
                raise

        return temp

    def makesimplepred(self):
        def simple_predicate(c, escaped):
            return (not c.isspace()) or escaped
        return simple_predicate

    def simpleeoc(self, temp, escaped):
        # try to handle a "bad" escape gracefully
        if escaped: temp += "\\"

    def makesimpleescape(self):
        def simple_escape(c):
            if c.isspace() or c in ("\\", "$"):
                return c
            return "\\" + c
        return simple_escape

    def simple(self):
        pred = self.makesimplepred()
        eoc = self.simpleeoc
        escape = self.makesimpleescape()
        return self.parsewhile(pred, eoc, escape, False)

    def makequotedpred(self, q):
        def quoted_predicate(c, escaped):
            return c != q or escaped
        return quoted_predicate

    def quotedeoc(self, temp, escaped):
        raise UnexpectedEndOfCommand("unterminated quote")

    def makequotedescape(self, q):
        def quoted_escape(c):
            if c == q:
                return c
            elif c == "$":
                return c
            return "\\" + c
        return quoted_escape

    def quoted(self):
        q = self.c()
        self.n()

        pred = self.makequotedpred(q)
        eoc = self.quotedeoc
        escape = self.makequotedescape(q)
        return self.parsewhile(pred, eoc, escape, True)

    def c(self):
        return self.command[self.offset]

    def n(self):
        self.checkeoc()
        self.offset += 1

    def iseoc(self):
        return self.offset+1 >= len(self.command)

    def checkeoc(self):
        if self.iseoc():
            raise EndOfCommandException()

