#
# ginvoke: a programmable launcher for GTK+
# Copyright (C) 2011 Thomas Lee
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import gtk

import os
import re

from signal import signal, SIG_IGN, SIG_DFL, SIGCHLD, SIGTERM, SIGQUIT, SIGINT

from .engine import BadCommandException

ALL = {}

def action(fn):
    ALL[fn.__name__] = fn
    return fn

@action
def run(context, args):
    if os.fork() == 0:
        context.server.stop()
        os.setsid()
        os.umask(0)
        if os.fork(): os._exit(0)
        signal(SIGCHLD, SIG_IGN)
        signal(SIGTERM, SIG_DFL)
        signal(SIGINT, SIG_DFL)
        signal(SIGQUIT, SIG_DFL)
        os.execvp(args[0], filter(None, args))

@action
def quit(context, args):
    gtk.main_quit()

@action
def url(context, args):
    run(context, (context.settings.browser,) + args)

@action
def reconfigure(context, args):
    context.settings.load()

MATH_REGEX = re.compile(r'^\s*[0-9][0-9\+\-/\*% ]+\s*$')

@action
def wildcard(context, args):
    command = context.command
    if command.startswith("http://") or command.startswith("https://"):
        url(context, (command,))
    elif re.match(r'^.*\.(com|org|net|us|ws|co|com\.au|org\.au|net\.au)$', command.strip()):
        url(context, ("http://%s" % command,))
    elif MATH_REGEX.search(command + ' '.join(context.args)):
        maths = command + ' '.join(context.args)
        context.window.entry.set_text(str(eval(maths)))
        context.window.entry.select_region(0, -1)
        return True
    elif len(context.rawargs) == 2 and context.rawargs[1].startswith("$"):
        context.window.entry.set_text(command)
        context.window.entry.select_region(0, -1)
        return True
    elif executable_exists(context, command):
        run(context, (command,) + tuple(args))
    else:
        raise BadCommandException("unknown command: %s" % command)

def executable_exists(context, command):
    foundexe = executable(command)
    if not foundexe:
        for path in context["env"]["PATH"].split(os.pathsep):
            full_path = os.path.join(path, command)
            if executable(full_path):
                foundexe = True
                return foundexe
    return foundexe

def executable(exe):
    # TODO symlinks will probably fuck this up
    return os.path.exists(exe) and not os.path.isdir(exe) and os.access(exe, os.X_OK)

