#!/usr/bin/env python
# coding: utf-8

import os, os.path, sys, re, zlib, glob, struct
from pack_file import PackFile
from obj_id import ObjId

def empty_if_none_deko(org_meth):
    def wrapper(gitdiscribe_ob):
        return '' if gitdiscribe_ob._tag is None else org_meth(gitdiscribe_ob)
    return wrapper

class Gitdiscribe(object):

    def __init__(self, directory='', debug=False):
        self.directory = directory
        self._debug = debug
        self.gitdir = os.path.join(directory, '.git')
        self.obdir = os.path.join(self.gitdir, 'objects')
        if not os.path.exists(directory):
            print "'{0}' does not exists".format(directory)
            sys.exit(1)
        elif not os.path.exists(self.gitdir):
            print "'{0}' is not a Git repo".format(directory)
            # sys.exit(1)
            self._tag = None
            return None
        self.pack = PackFile(self.gitdir, debug=debug)
        self.get_packed_refs()
        self.get_head()
        self.get_tags()
        if not self.tags:
            self._tag = None
            return None
        top = self.head
        parents = [None]
        dist = 0
        while True:
            parents = self.get_parents(ObjId(top))
            if top in self.tags or len(parents) == 0:
                break
            top = parents[0] # only look at first parent like in git log --first-parent (see http://wp.me/pqI3H-eU)
            dist += 1
            self.prtdb(parents)

        self._tag = {'name': self.tags.get(top), 'dist': dist, 'commit': self.head} if self.tags.get(top) else None

    @property
    @empty_if_none_deko
    def tag(self):
        if self._tag['dist'] == 0:
            out = self.tag_short
        else:
            out = self.tag_long
        return out

    @property
    @empty_if_none_deko
    def tag_long(self):
        t = self._tag
        return '{0}-{1}-g{2}'.format(t['name'], t['dist'], t['commit'][0:8])

    @property
    @empty_if_none_deko
    def tag_short(self):
        return self._tag['name']

    @property
    @empty_if_none_deko
    def tag_number(self):
        return re.sub('^\D+', '', self.tag, re.IGNORECASE)

    @property
    @empty_if_none_deko
    def tag_number_short(self):
        return re.sub('^\D+', '', self.tag_short, re.IGNORECASE)

    @property
    @empty_if_none_deko
    def tag_number_long(self):
        return re.sub('^\D+', '', self.tag_long, re.IGNORECASE)

    def write_version_file(self, path='version.py'):
        if not self.tag:
            return
        t = self._tag
        with open(path, 'w') as fh:
            fh.write('''# written by gitdiscribe
FULL_VERSION = "{0}"
VERSION = "{1}"
GIT_REVISION = "{2}"'''.format(self.tag_number, self.tag_number_short, t['commit'][0:8]))

    def get_packed_refs(self):
        self.packed_refs = {}
        pr_path = os.path.join(self.gitdir, 'packed-refs')
        if os.path.exists(pr_path):
            with open(pr_path, 'r') as f:
                for l in f:
                    mo = re.match('^(\S+) (.+)$', l)
                    if mo:
                        self.packed_refs[mo.group(2)] = mo.group(1)


    def get_tags(self):
        tags = {}
        tagdir = os.path.join(self.gitdir, 'refs', 'tags')
        for tag in os.listdir(tagdir):
            obj_sha = self.read_oneliner(os.path.join(tagdir, tag))
            tags[tag] = obj_sha
        for ref, obj_sha in self.packed_refs.items():
            mo = re.match('refs/tags/(.+)$', ref)
            if mo:
                tags[mo.group(1)] = obj_sha
        if len(tags) == 0:
            self.tags = None
            return False
        self.tags = {}
        for tag, obj_sha in tags.items():
            blines = self.obj_as_lines(ObjId(obj_sha))
            mo = re.match(r'tag \d+\x00object (\S+)', blines[0])
            # annotated tags point to a tag obj while leightweights directly point to a commit obj
            c_sha = mo.group(1) if mo else obj_sha
            self.tags[c_sha] = tag
        self.prtdb(self.tags)

    def get_head(self):
        head = open(os.path.join(self.gitdir, 'HEAD')).read()
        self.prtdb(head)
        ref = re.match(r'ref: (\S+)', head).group(1)
        self.prtdb(ref)
        # self.head = ObjId(self.get_ref(ref))
        self.head = self.get_ref(ref)
        self.prtdb(self.head)

    def get_ref(self, ref):
        ref_path = os.path.join(self.gitdir, ref)
        if ref in self.packed_refs:
            return self.packed_refs[ref]
        elif os.path.exists(ref_path):
            return self.read_oneliner(ref_path)
        else:
            return None

    def read_oneliner(self, path):
        return open(path).read().strip()

    def get_parents(self, obj_id):
        clines = self.obj_as_lines(obj_id)
        return [ mo.group(1) for mo in [re.match('parent (\S+)', l) for l in clines] if mo ]

    def obj_as_lines(self, sha):  # roughly like: git cat-file -p <sha1>
        fs_path = os.path.join(self.obdir, sha.d, sha.f)
        if os.path.exists(fs_path):
            self.prtdb('  -- read from file {0}'.format(sha))
            obj =  zlib.decompress(open(fs_path).read())
        # elif self.pack: # python 3 only
        elif self.pack.__bool__():
            self.prtdb('  try read  from packfile {0}'.format(sha))
            obj = self.pack.read_object(sha)
        else:
            self.prtdb('  no Way! {0}'.format(sha))
            obj = ''
        return re.split(r'(\r\n|\n|\r)', obj)

    def prtdb(self, matter):
        if self._debug:
            print(matter)

if  __name__ == '__main__':

    path = '..'
    for d in os.listdir(path):
        dp = os.path.join(path, d)
        if os.path.isdir(dp):
            giti = Gitdiscribe(dp)
            print('{0}: {1}'.format(d, giti.tag))
            # print(giti.tag)
            # print(giti.tag_short)
            # print(giti.tag_long)
            # print(giti.tag_number)
            # print(giti.tag_number_short)
            # print(giti.tag_number_long)
