#!/usr/bin/env python
# encoding: utf-8
'''
bin.check_log -- shortdesc

bin.check_log is a description

It defines classes_and_methods

@author:     Shawn Crosby
            
@copyright:  2014 Salesforce.com. All rights reserved.
            
@license:    license

@contact:    scrosby@salesforce.com
@deffield    updated: 2014-07-16
'''

import sys
import os
import requests
from gus.BacklogClient import BacklogClient
import datetime, re
from optparse import OptionParser
from ws.releases.Releases import Releases

__all__ = []
__version__ = 0.1
__date__ = '2014-07-16'
__updated__ = '2014-07-16'

DEBUG = 0
TESTRUN = 0
PROFILE = 0

def main(argv=None):
    '''Command line options.'''
    
    program_name = os.path.basename(sys.argv[0])
    program_version = "v0.1"
    program_build_date = "%s" % __updated__
 
    program_version_string = '%%prog %s (%s)' % (program_version, program_build_date)
    #program_usage = '''usage: spam two eggs''' # optional - will be autogenerated by optparse
    program_longdesc = '''''' # optional - give further explanation about what the program does
    program_license = "Copyright 2014 user_name (organization_name)                                            \
                Licensed under the Apache License 2.0\nhttp://www.apache.org/licenses/LICENSE-2.0"
 
    if argv is None:
        argv = sys.argv[1:]
    try:
        # setup option parser
        parser = OptionParser(version=program_version_string, epilog=program_longdesc, description=program_license)
        parser.add_option("-s", "--start", dest="start", action="store", help="Beginning Date Range [default: %default]")
        parser.add_option("-e", "--end", dest="end", action="store", help="Ending Date Range [default: %default]")
        parser.add_option("-m", "--masterversion", dest="master", action="store", help="Version currently in Master")
        parser.add_option("-r", "--maintversion", dest="maint", action="store", help="Version currently in Release Branch")
        
        # set defaults
        today = datetime.date.today()
        tomorrow = today + datetime.timedelta(days=1)
        yesterday = today - datetime.timedelta(days=1)
        
        # figure out the current release #'s
        r = Releases(today)

        parser.set_defaults(start=today, end=tomorrow, maint=r.maint, master=r.master)
        
        # process options
        (opts, args) = parser.parse_args(argv)
        
        start = opts.start
        end = opts.end
        maint = opts.maint
        master = opts.master
        
        if len(args) == 1 and args[0] == 'yesterday':
            start = yesterday
            end = today
            r = Releases(yesterday)
            maint = r.maint
            master = r.master
        
        if opts.master is None:
            raise Exception("You must specify a version # for the master branch")
        
        if opts.maint is None:
            raise Exception("You must specify a version # for the release branch")
        
        # MAIN BODY #
        gus = BacklogClient()
        
        r = requests.get('https://warm-ravine-1655.herokuapp.com/commits?start={}&end={}'.format(start, end), headers={'content-type':'application/json'})
        commits = r.json()
        no_gus=0
        no_version=0
        
        print("|".rjust(126, '-'))
        print("{}{}{}{}{}{}".format("GUS ID".rjust(15), "REPO".rjust(30), "COMMITTER".rjust(20), "BRANCH".rjust(20), "BUILD".rjust(20), "STATUS".rjust(20)))
        print("|".rjust(126, '-'))
        for commit in commits:
            version = commit['version']
            if commit['branch'] == 'master':
                version = master
            elif commit['branch'].endswith('_release'):
                version = maint
            
            if version is not None:
                    
                if commit['gus_id'] is not None:
                    gus_id = commit['gus_id']
                    if "-" not in gus_id:
                        gus_id = str(gus_id).replace("W", "W-").replace("w", "W-")
                        
                    try:   
                        work = gus.find_work(gus_id)
                        build_name = ""
                        if work is not None:
                            buildid = work['Scheduled_Build__c']
                            if buildid is not None:
                                build = gus.get_build_record(buildid)  
                                if build is not None:
                                    build_name = build['Name']
                                    build_version = get_version_from_target(build_name)
                                    tag = ""
                                    if version != build_version:
                                        tag = " *[{}]".format(version)
                        committer = bytes.decode(commit['committer'].encode('ascii','replace'))
                        print("{}{}{}{}{}{}{}".format(gus_id.rjust(15)[:15], commit['repo'].rjust(30)[:30], committer.rjust(20)[:20], commit['branch'].rjust(20)[:20], build_name.rjust(20)[:20], work['Status__c'].rjust(20)[:20], tag))
                    except Exception as e:
                        print("Fake Work ID: {}".format(commit['gus_id']))
                else:
                    no_gus += 1 

            else:
                no_version += 1

                        
        print("Out of {} commits pushed to versioned branches, {}% of commits had no gus_id".format(len(commits)-no_version, round(no_gus/len(commits)*100,2)))
        print("There were {} commits pushed to dev branches".format(no_version))
    except Exception as e:
        #raise e
        indent = len(program_name) * " "
        sys.stderr.write(program_name + ": " + repr(e) + "\n")
        sys.stderr.write(indent + "  for help use --help")
        return 2
    
def get_version_from_target(string):
    match = re.search("MC_(\d{3}(\.\d)?)", string)
    if match is not None:
        out = match.group(1)
        return out
    return None

if __name__ == "__main__":
    if DEBUG:
        sys.argv.append("-h")
    if TESTRUN:
        import doctest
        doctest.testmod()
    if PROFILE:
        import cProfile
        import pstats
        profile_filename = 'bin.check_log_profile.txt'
        cProfile.run('main()', profile_filename)
        statsfile = open("profile_stats.txt", "wb")
        p = pstats.Stats(profile_filename, stream=statsfile)
        stats = p.strip_dirs().sort_stats('cumulative')
        stats.print_stats()
        statsfile.close()
        sys.exit(0)
    sys.exit(main())