#!/usr/bin/env python
# encoding: utf-8
'''
bin.check_log -- shortdesc

bin.check_log is a description

It defines classes_and_methods

@author:     Shawn Crosby
            
@copyright:  2014 Salesforce.com. All rights reserved.
            
@license:    license

@contact:    scrosby@salesforce.com
@deffield    updated: 2014-07-16
'''

import sys
import os
from gus.BacklogClient import BacklogClient
import datetime, re
from optparse import OptionParser
from ws.commits.Commits import Commits

__all__ = []
__version__ = 0.1
__date__ = '2014-07-16'
__updated__ = '2014-07-16'

DEBUG = 0
TESTRUN = 0
PROFILE = 0

def main(argv=None):
    '''Command line options.'''
    
    program_name = os.path.basename(sys.argv[0])
    program_version = "v0.1"
    program_build_date = "%s" % __updated__
 
    program_version_string = '%%prog %s (%s)' % (program_version, program_build_date)
    #program_usage = '''usage: spam two eggs''' # optional - will be autogenerated by optparse
    program_longdesc = '''''' # optional - give further explanation about what the program does
    program_license = "Copyright 2014 user_name (organization_name)                                            \
                Licensed under the Apache License 2.0\nhttp://www.apache.org/licenses/LICENSE-2.0"
 
    if argv is None:
        argv = sys.argv[1:]
    try:
        # setup option parser
        parser = OptionParser(version=program_version_string, epilog=program_longdesc, description=program_license)
        parser.add_option("-s", "--start", dest="start", action="store", help="Beginning Date Range [default: %default]")
        parser.add_option("-e", "--end", dest="end", action="store", help="Ending Date Range [default: %default]")
        
        # set defaults
        today = datetime.date.today()
        tomorrow = today + datetime.timedelta(days=1)
        yesterday = today - datetime.timedelta(days=1)
        
        parser.set_defaults(start=today, end=tomorrow)
        
        # process options
        (opts, args) = parser.parse_args(argv)
        
        start = opts.start
        end = opts.end
        
        if len(args) == 1 and args[0] == 'yesterday':
            start = yesterday
            end = today
        
        # MAIN BODY #
        gus = BacklogClient()
        c = Commits()
        gus_ids = c.get_unique_gus_ids(start, end)
        
        print("{}|".format("-" * 125))
        print("{}{}{}{}{}".format("GUS_ID".rjust(10), "BRANCHES".rjust(50), "COMMITTER".rjust(25), "TARGET".rjust(15), "STATUS".rjust(20)))
        print("{}|".format("-" * 125))
        
        for gus_id in gus_ids:
            branches = c.get_branches_for_gus_id(gus_id)
            branch_list = []
            try:
                work = gus.find_work(gus_id)
                if work['Scheduled_Build__c'] is not None:
                    build = gus.get_build_record(work['Scheduled_Build__c'])
                    build_name = build['Name']
                    status = work['Status__c']
                for branch in branches:
                    branch_commits = c.get_commits_for_gus_id_on_branch(gus_id, branch)
                    branch_list.append("{} ({})".format(branch, len(branch_commits)))
                    
                committers = c.get_committers_for_gus_id(gus_id)
                if len(committers) > 1:
                    committer = "{} (+{})".format(committers[0][:15], len(committers) - 1)
                else:
                    committer = committers[0][:20]
                    
                print("{}{}{}{}{}".format(gus_id.rjust(10), ', '.join(branch_list).rjust(50), committer.rjust(25), build_name.rjust(15), status.rjust(20)[:20]))
            except:
                print("Invalid Gus ID: {}".format(gus_id))
    except Exception as e:
        raise e
        indent = len(program_name) * " "
        sys.stderr.write(program_name + ": " + repr(e) + "\n")
        sys.stderr.write(indent + "  for help use --help")
        return 2
    
if __name__ == "__main__":
    if DEBUG:
        sys.argv.append("-h")
    if TESTRUN:
        import doctest
        doctest.testmod()
    if PROFILE:
        import cProfile
        import pstats
        profile_filename = 'bin.check_log_profile.txt'
        cProfile.run('main()', profile_filename)
        statsfile = open("profile_stats.txt", "wb")
        p = pstats.Stats(profile_filename, stream=statsfile)
        stats = p.strip_dirs().sort_stats('cumulative')
        stats.print_stats()
        statsfile.close()
        sys.exit(0)
    sys.exit(main())