import os
import json
import datetime
import subprocess

import yaml
import tornado.web
import tornado.ioloop


class Processor(object):
    def __init__(self, application, request):
        self.application = application
        self.request = request

        try:
            self.event = request["headers"]["X-GitHub-Event"]
            self.branch = request["payload"]["ref"].split("/")[-1]
        except Exception, ex:
            print "ERROR:", ex
            self.application.ioloop.add_timeout(self.application.interval, self.application.next)
        else:
            self.tasks = ["git checkout %s" % self.branch, "git pull origin %s" % self.branch, self.read_config]
            self.process = None
            self.application.ioloop.add_timeout(self.application.interval, self.next)

    def read_config(self):
        with open(self.application.config_filename) as config_file:
            config = yaml.safe_load(config_file)

        try:
            tasks = config[self.branch][self.event]
            assert isinstance(tasks, list)
        except (KeyError, AssertionError):
            pass
        else:
            self.tasks.extend(tasks)

    def next(self):
        try:
            task = self.tasks.pop(0)
        except IndexError:
            self.application.ioloop.add_timeout(self.application.interval, self.application.next)
        else:
            if hasattr(task, "__call__"):
                task()
                self.application.ioloop.add_timeout(self.application.interval, self.next)
            else:
                self.process = subprocess.Popen(task, shell=True, stdout=subprocess.PIPE)
                self.application.ioloop.add_timeout(self.application.interval, self.wait)

    def wait(self):
        return_code = self.process.returncode
        if return_code is None:
            self.application.ioloop.add_timeout(self.application.interval, self.wait)
        elif return_code == 0:
            self.application.ioloop.add_timeout(self.application.interval, self.next)
        else:
            #TODO: mark bad build
            self.application.ioloop.add_timeout(self.application.interval, self.application.next)


class Application(tornado.web.Application):
    def __init__(self, config_filename="ci.yml", interval=1):
        super(Application, self).__init__([(".*", Handler)])
        self.queue = []
        self.current = None

        self.ioloop = tornado.ioloop.IOLoop.instance()

        self.config_filename = config_filename
        self.interval = datetime.timedelta(seconds=interval)

        self.ioloop.add_timeout(self.interval, self.next)

    def next(self):
        try:
            request = self.queue.pop(0)
        except IndexError:
            self.current = None
        else:
            self.current = Processor(self, request)


class Handler(tornado.web.RequestHandler):
    def post(self):
        try:
            request = {
                "headers": self.request.headers,
                "payload": json.loads(self.request.payload)
            }
        except ValueError:
            raise tornado.web.HTTPError(404)
        else:
            self.application.queue.append(request)


if "__main__" == __name__:
    import argparse

    parser = argparse.ArgumentParser(prog="githubci",
                                     description="A simple realization of Continuous Integration for GitHub")
    parser.add_argument("repository", type=str, help="path to repository")
    parser.add_argument("-i", "--interface", type=str, help="listen interface", default="127.0.0.1")
    parser.add_argument("-p", "--port", type=int, help="listen port", default=9092)
    parser.add_argument("-c", "--config", type=str, help="configuration filename", default="ci.yml")

    args = parser.parse_args()

    os.chdir(args.repository)

    app = Application(config_filename=args.config)
    app.listen(args.port, args.interface)
    app.ioloop.start()