__author__ = 'ardevelop'

import time
import os
import json
import datetime
import subprocess

import yaml
import tornado.web
import tornado.ioloop


EVENT_PUSH = "push"
EVENT_RELEASE = "release"
EVENTS = {EVENT_PUSH, EVENT_RELEASE}

STATUS_WAITING = 0
STATUS_OK = 1
STATUS_FAILED = 2


class WebHook(object):
    def __new__(cls, application, request):
        user_agent = request.headers.get("User-Agent", None)
        if user_agent:
            if user_agent.startswith("GitHub"):
                return GitHubWebHook(application, request)

            raise Exception("User agent not supported: %s" % user_agent)

        raise Exception("User agent not specified")

    def __init__(self, application, request):
        self.application = application
        self.application.enqueue(self.process)
        self.status = STATUS_WAITING
        self.subprocess = None

        self.created = time.time()
        self.started = None
        self.completed = None

        self.event = None
        self.actor = None
        self.branch = None
        self.commit = None
        self.tasks = None

    def get_tasks(self):
        return ["git checkout .", "git checkout %s" % self.branch, "git pull origin %s" % self.branch,
                "git checkout %s" % self.commit, self.extend]

    def start(self):
        self.started = time.time()
        try:
            assert self.event in EVENTS
            assert self.actor is not None
            assert self.branch is not None
            assert self.commit is not None
            self.tasks = self.get_tasks()
        except Exception, ex:
            self.fail(ex)
        else:
            self.application.enqueue(self.process)

    def process(self):
        try:
            task = self.tasks.pop(0)
        except IndexError:
            self.done()
        else:
            if hasattr(task, "__call__"):
                task()
            else:
                try:
                    self.subprocess = subprocess.Popen(task, shell=True)
                except OSError, ex:
                    self.fail(ex)
                else:
                        self.application.enqueue(self.wait_subprocess)

    def fail(self, reason):
        self.completed = time.time()
        self.status = STATUS_FAILED

    def done(self):
        self.completed = time.time()
        self.status = STATUS_OK
        self.application.enqueue(self.application.next)

    def wait_subprocess(self):
        pid, status = os.waitpid(self.subprocess.pid, os.WNOHANG)
        if pid == 0:
            self.application.enqueue(self.wait_subprocess)
        else:
            if status == 0:
                self.application.enqueue(self.process)
            else:
                self.fail(status)

    def extend(self):
        if os.path.isfile(self.application.config_filename):
            try:
                with open(self.application.config_filename) as config_file:
                    config = yaml.safe_load(config_file)

                try:
                    custom = config[self.branch][self.event]
                except KeyError:
                    try:
                        custom = config["*"][self.event]
                    except KeyError:
                        custom = None

                if custom:
                    self.tasks.extend(custom)
                    return self.application.enqueue(self.process)
            except Exception, ex:
                self.fail(ex)

        self.done()


class GitHubWebHook(WebHook):
    def __new__(cls, application, request):
        event = request.headers["X-GitHub-Event"]
        if event == "push":
            return GitHubPushEvent(application, request)
        elif event == "release":
            return GitHubReleaseEvent(application, request)
        else:
            raise Exception("Unsupported event: %s" % event)

    def __init__(self, application, request):
        super(GitHubWebHook, self).__init__(application, request)
        self.payload = json.loads(request.body)


class GitHubPushEvent(GitHubWebHook):
    def __init__(self, application, request):
        super(GitHubPushEvent, self).__init__(application, request)

        self.event = EVENT_PUSH
        self.branch = self.payload["ref"].split("/")[-1]
        self.commit = self.payload["after"]
        self.actor = self.payload["pusher"]["name"]


class GitHubReleaseEvent(GitHubWebHook):
    def __init__(self, application, request):
        super(GitHubReleaseEvent, self).__init__(application, request)

        self.event = EVENT_RELEASE


class Application(tornado.web.Application):
    def __init__(self, config_filename="ci.yml", interval=1):
        super(Application, self).__init__([(".*", Handler)])
        self.queue = []
        self.done = []

        self.ioloop = tornado.ioloop.IOLoop.instance()

        self.config_filename = config_filename
        self.interval = datetime.timedelta(seconds=interval)

        self.enqueue(self.process)

    def enqueue(self, function):
        self.ioloop.add_timeout(self.interval, function)

    def process(self):
        try:
            hook = self.queue.pop(0)
        except IndexError:
            self.enqueue(self.process)
        else:
            hook.start()


class Handler(tornado.web.RequestHandler):
    def get(self):
        self.write("<pre>")
        for job in self.application.done:
            self.write("%25s %25s %25s\n" % (job.branch, job.event, job.status))
        self.write("</pre>")

    def post(self):
        try:
            hook = WebHook(self.application, self.request)
            if hook:
                self.application.queue.append(hook)
                self.application.done.append(hook)
        except Exception, ex:
            raise tornado.web.HTTPError(400, str(ex))


if "__main__" == __name__:
    app = Application(config_filename="ci.yml")
    app.listen(8080)
    app.ioloop.start()