__author__ = 'ardevelop'

import os
import json
import datetime
import subprocess
import shutil

import sqlite3
import yaml
import tornado.web
import tornado.ioloop


EVENT_PUSH = "push"

STATUS_WAITING = 0
STATUS_OK = 1
STATUS_FAILED = 2


class SqliteWrapper(object):
    def __init__(self, path):
        self._connection = sqlite3.connect(path)
        self._connection.row_factory = sqlite3.Row

    def execute(self, statement, *args):
        cursor = self._connection.cursor()
        cursor.execute(statement, args)
        return self._connection, cursor

    def get(self, statement, *args):
        con, cur = self.execute(statement, *args)
        return cur.fetchone() if cur.rowcount > 0 else None

    def all(self, statement, *args):
        con, cur = self.execute(statement, *args)
        return cur.fetchall()

    def new(self, statement, *args):
        con, cur = self.execute(statement, *args)
        con.commit()
        return cur.lastrowid

    def update(self, statement, *args):
        con, cur = self.execute(statement, *args)
        con.commit()
        return True


class Worker(object):
    def __init__(self, application):
        self.application = application
        application.enqueue(self._work)

        self.task = None
        self.subprocess = None
        self.commands = None

    def _work(self):
        task = self.application.db.get("SELECT * FROM task LIMIT 1")
        if task:
            push = self.task["push_id"]
            branch = self.task["branch_name"]
            commit = self.task["push_commit"]

            self.commands = ["git checkout .", "git checkout %s" % branch, "git pull origin %s" % branch,
                             "git checkout %s" % commit, self._get_local_commands]

            self.application.db.udpate("UPDATE push SET started = DATETIME('NOW') WHERE id = ?", push)
            self.application.enqueue(self._next_command)
        else:
            self.application.enqueue(self._work)

    def _get_local_commands(self):
        if os.path.isfile(self.application.config_filename):
            try:
                with open(self.application.config_filename) as config_file:
                    config = yaml.safe_load(config_file)

                branch = self.task["branch_name"]

                try:
                    local = config[branch]
                except KeyError:
                    try:
                        local = config["*"]
                    except KeyError:
                        local = None

                if local:
                    self.commands.extend(local)
                    return self.application.enqueue(self._next_command)
            except Exception, ex:
                self._fail(ex)

        self._success()

    def _next_command(self):
        try:
            command = self.commands.pop(0)
        except IndexError:
            self._success()
        else:
            if hasattr(command, "__call__"):
                command()
            else:
                try:
                    self.subprocess = subprocess.Popen(command, shell=True)
                except OSError, ex:
                    self._fail(ex)
                else:
                    self.application.enqueue(self._wait_subprocess)

    def _wait_subprocess(self):
        pid, status = os.waitpid(self.subprocess.pid, os.WNOHANG)
        if pid == 0:
            self.application.enqueue(self._wait_subprocess)
        else:
            if status == 0:
                self.application.enqueue(self._next_command)
            else:
                self._fail(status)

    def _complete(self):
        self.task = None
        self.subprocess = None
        self.commands = None
        self.application.enqueue(self._work)

    def _fail(self, reason=None):
        self.application.db.udpate("UPDATE push SET status = ?, completed = DATETIME('NOW') WHERE id = ?",
                                   STATUS_FAILED, self.task["push_id"])
        self._complete()

    def _success(self):
        self.application.db.udpate("UPDATE push SET status = ?, completed = DATETIME('NOW') WHERE id = ?",
                                   STATUS_OK, self.task["push_id"])
        self._complete()


class WebHook(object):
    @staticmethod
    def create(application, request):
        user_agent = request.headers.get("User-Agent", None)
        if user_agent:
            if user_agent.startswith("GitHub"):
                return GitHubWebHook.create(application, request)

            raise Exception("User agent not supported: %s" % user_agent)

        raise Exception("User agent not specified")

    def __init__(self, application):
        self.application = application

    def get_repository_url(self):
        raise NotImplementedError()

    def get_repository(self):
        raise NotImplementedError()

    def get_branch(self):
        raise NotImplementedError()

    def get_commit(self):
        raise NotImplementedError()

    def push(self):
        repository_url = self.get_repository_url()
        repository_id = self.application.db.get("SELECT id FROM repository WHERE url = ?", repository_url)
        if not repository_id:
            repository_url, repository_owner, repository_name = self.get_repository()
            repository_id = self.application.db.new("INSERT INTO repository VALUES (NULL, ?, ?, ?, DATETIME('NOW'))",
                                                    repository_url, repository_owner, repository_name)

        branch = self.get_branch()
        branch_id = self.application.db.get("SELECT id FROM branch WHERE repository = ? AND name = ?",
                                            repository_id, branch)
        if not branch_id:
            branch_id = self.application.db.new("INSERT INTO branch VALUES (NULL, ?, ?, NULL, NULL, NULL)",
                                                branch, repository_id)

        commit, author = self.get_commit()

        self.application.db.new("INSERT INTO push VALUES (NULL, ?, ?, ?, ?, DATETIME('NOW'), NULL, NULL)",
                                branch_id, commit, author, STATUS_WAITING)


class GitHubWebHook(WebHook):
    @staticmethod
    def create(application, request):
        event = request.headers["X-GitHub-Event"]
        if event == "push":
            return GitHubPushEvent(application, request)
        else:
            raise Exception("Unsupported event: %s" % event)

    def __init__(self, application, request):
        super(GitHubWebHook, self).__init__(application)
        self.payload = json.loads(request.body)


class GitHubPushEvent(GitHubWebHook):
    def __init__(self, application, request):
        super(GitHubPushEvent, self).__init__(application, request)

        self.event = EVENT_PUSH
        self.branch = self.payload["ref"].split("/")[-1]
        self.commit = self.payload["after"]
        self.actor = self.payload["pusher"]["name"]

    def get_repository_url(self):
        return self.payload["repository"]["url"]

    def get_repository(self):
        repository = self.payload["repository"]
        return repository["url"], repository["name"], repository["owner"]["name"]

    def get_branch(self):
        return self.payload["ref"].split("/")[-1]

    def get_commit(self):
        return self.payload["after"], self.payload["pusher"]["name"]


class Application(tornado.web.Application):
    def __init__(self, database="/tmp/ci.sqlite", config_filename="ci.yml", interval=1):
        super(Application, self).__init__([(".*", Handler)])

        self.ioloop = tornado.ioloop.IOLoop.instance()
        self.config_filename = config_filename
        self.interval = datetime.timedelta(seconds=interval)

        if not os.path.exists(database):
            shutil.copyfile(os.path.abspath(os.path.join(__file__, "..", "db.sqlite")), database)

        self.db = SqliteWrapper(database)

        #TODO: add one worker per repository for parallel calculations
        self.workers = {
            "tmp": Worker(self)
        }

    def enqueue(self, function):
        self.ioloop.add_timeout(self.interval, function)


class Handler(tornado.web.RequestHandler):
    def get(self):
        self.write("<table border=1 cellpadding=5>")
        self.write("<tr><th>Status</th><th>Commit</th><th>Author</th><th>Completed</th><th>Duration</th></tr>")
        for push in self.application.db.all("SELECT * FROM push"):
            self.write("<tr><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td></tr>" %
                       (push["status"], push["commit"], push["author"], push["started"], push["completed"]))
        self.write("</table>")

    def json(self, obj, status=None):
        if status:
            self.set_status(status)

        self.set_header("Content-Type", "application/json")
        self.finish(json.dumps(obj))

    def post(self):
        try:
            hook = WebHook.create(self.application, self.request)
            if hook:
                hook.push()
                self.json({"success": True, "accepted": True})
            else:
                self.json({"success": True, "accepted": False})
        except Exception, ex:
            self.json({"success": False, "message": str(ex)}, 400)


if "__main__" == __name__:
    app = Application(config_filename="ci.yml")
    app.listen(8080)
    app.ioloop.start()