import re
from datetime import datetime
from subprocess import check_output


def _run_git(cmd, args=None):
    cmd_list = ['git', cmd]
    if args:
        for arg in args:
            cmd_list.append(arg)
    return check_output(cmd_list)


def _parse_log(raw):
    commit = {}
    regex = re.compile(r'(?:commit\s*(?P<SHA1>[a-z0-9]{40}))|'
                       r'(?:Author:\s*(?P<AUTHOR>.+)\s+)|'
                       r'(?:Date:\s*(?P<DATE>[0-9:\+\- ]+))|'
                       r'(?:\s*(?P<EMPTYLINE>\n))|'
                       r'(?:\s*(?P<TEXT>.+\n))')

    for match in regex.finditer(raw):
        typ = match.lastgroup
        # When a new SHA1 appears one commit has been finished.
        if typ == 'SHA1':
            if commit:
                commit['text'] = commit['text'].strip()
                yield commit
            commit = {}
            commit['sha1'] = match.group('SHA1')
        elif typ == 'AUTHOR':
            commit['author'] = match.group('AUTHOR')
        elif typ == 'DATE':
            commit['date'] = datetime.strptime(match.group('DATE'), '%Y-%m-%d %H:%M:%S %z')
        # Commit description follows the dateline, so let's
        # append text and whitespace after parsing the dateline.
        elif ('date' in commit) and (typ == 'TEXT' or typ == 'EMPTYLINE'):
            if not 'text' in commit:
                commit['text'] = ''
            commit['text'] += (match.group('TEXT') or match.group('EMPTYLINE'))

    commit['text'] = commit['text'].strip()
    yield commit  # Do not forget the inital commit!


def _parse_status(raw):
    st = {}
    # specification see 'man git status'
    regex = re.compile(r'(?P<ST>[MADRCU\s\?]{2})\s(?P<FILE>.*)')
    for match in regex.finditer(raw):
        st[match.group('FILE')] = match.group('ST')
    return st
