# -*- coding: utf-8 -*-

import socket
import errno
from kola.utils import safe_call


class GStream(object):
    """
    为了和tornado的IOStream接口匹配
    """

    _closed = None
    rfile = None
    wfile = None
    on_close_callback = None

    def __init__(self, sock):
        self.sock = sock
        self.rfile = sock.makefile('rb')
        self.wfile = sock.makefile('wb', 0)
        self._closed = False

    def close(self, exc_info=False):
        if self._closed:
            # 如果已经关闭过，就直接返回了
            return

        self._closed = True

        if self.rfile:
            safe_call(self.rfile.close)
            self.rfile = None

        if self.wfile:
            safe_call(self.wfile.close)
            self.wfile = None

        if self.sock:
            safe_call(self.sock.close)
            self.sock = None

    def set_close_callback(self, callback):
        """
        设置回调
        """
        self.on_close_callback = callback

    def readline(self, on_read_complete):
        """
        接收消息
        """

        data = self.rfile.readline()
        if not data:
            # 重复调用也没关系
            self.close()

            if self.on_close_callback:
                safe_call(self.on_close_callback)
            return

        safe_call(on_read_complete, data)

    def write(self, data, callback=None):
        """
        写数据
        """

        if self.wfile:
            safe_call(self.wfile.write, data)
            safe_call(callback)

    def closed(self):
        return self._closed

    def writing(self):
        """
        不太好实现，就统一用False吧
        对于短链接可能会有点影响
        """
        return False

    def set_nodelay(self, value):
        """
        直接抄的tornado
        """
        if (self.sock is not None and
                    self.sock.family in (socket.AF_INET, socket.AF_INET6)):
            try:
                self.sock.setsockopt(socket.IPPROTO_TCP,
                                     socket.TCP_NODELAY, 1 if value else 0)
            except socket.error as e:
                # Sometimes setsockopt will fail if the socket is closed
                # at the wrong time.  This can happen with HTTPServer
                # resetting the value to false between requests.
                if e.errno not in (errno.EINVAL, errno.ECONNRESET):
                    raise