# -*- coding: utf-8 -*-

import socket
from kola import safe_call, logger


class Stream(object):
    """
    为了和tornado的IOStream接口匹配
    """

    rfile = None
    wfile = None
    on_close_callback = None

    def __init__(self, sock):
        self.sock = sock
        self.rfile = sock.makefile('rb')
        self.wfile = sock.makefile('wb', 0)

    def close(self, exc_info=False):
        if self.closed():
            # 如果已经关闭过，就直接返回了
            return

        if self.rfile:
            safe_call(self.rfile.close)
            self.rfile = None

        if self.wfile:
            safe_call(self.wfile.close)
            self.wfile = None

        if self.sock:
            safe_call(self.sock.close)
            self.sock = None

    def shutdown(self, how=2):
        """
        gevent的close只是把sock替换为另一个类的实例。
        这个实例的任何方法都会报错，但只有当真正调用recv、write或者有recv or send事件的时候，才会调用到这些函数，才可能检测到。
        而我们在endpoint对应的函数里spawn_later一个新greenlet而不做join的话，connection的while循环此时已经开始read了。

        之所以不把这个函数实现到connection，是因为shutdown更类似于触发一个close的事件
        用shutdown可以直接触发. how: 0: SHUT_RD, 1: SHUT_WR, else: all
        shutdown后还是会触发close事件以及相关的回调函数，不必担心
        """
        if self.closed():
            return
        safe_call(self.sock.shutdown, how)

    def set_close_callback(self, callback):
        """
        设置回调
        """
        self.on_close_callback = callback

    def read_until(self, terminator, on_read_complete):
        """
        为了兼容调用的方法
        """
        if terminator != '\n':
            logger.error('only support \\n as terminator for now, terminator: %s', terminator)
            raise TypeError('only support \\n as terminator for now')

        self._readline(on_read_complete)

    def write(self, data, callback=None):
        """
        写数据
        """

        if self.closed():
            return

        safe_call(self.wfile.write, data)
        safe_call(callback)

    def closed(self):
        return not self.sock

    def writing(self):
        """
        不太好实现，就统一用False吧
        对于短链接可能会有点影响
        """
        return False

    def set_nodelay(self, value):
        """
        直接抄的tornado
        """
        if (self.sock is not None and
                    self.sock.family in (socket.AF_INET, socket.AF_INET6)):
            safe_call(self.sock.setsockopt,
                      socket.IPPROTO_TCP, socket.TCP_NODELAY, 1 if value else 0)

    def _readline(self, on_read_complete):
        """
        接收一行消息
        """

        if self.closed():
            return

        # 异常也认为服务器断掉链接了，或者该断掉链接
        try:
            data = self.rfile.readline()
        except Exception, e:
            logger.exception('rfile.readline fail. e: %s', e)
            data = None

        if not data:
            # 重复调用也没关系
            self.close()

            if self.on_close_callback:
                safe_call(self.on_close_callback)
            return

        safe_call(on_read_complete, data)
