# -*- coding: utf-8 -*-

import functools
from kola import safe_call, logger


def count_reader(func):
    @functools.wraps(func)
    def func_wrapper(stream, *args, **kwargs):
        try:
            stream.reader_count += 1
            return func(stream, *args, **kwargs)
        finally:
            stream.reader_count -= 1

    return func_wrapper


def count_writer(func):
    @functools.wraps(func)
    def func_wrapper(stream, *args, **kwargs):
        try:
            stream.writer_count += 1
            return func(stream, *args, **kwargs)
        finally:
            stream.writer_count -= 1
    return func_wrapper


class Stream(object):
    """
    为了和tornado的IOStream接口匹配
    """

    ws = None
    _close_callback = None

    def __init__(self, ws):
        self.ws = ws

        self.reader_count = 0
        self.writer_count = 0

    def close(self, exc_info=False):
        if self.closed():
            # 如果已经关闭过，就直接返回了
            return

        safe_call(self.ws.close)
        self.ws = None

    def shutdown(self, how=2):
        """
        gevent的close只是把sock替换为另一个类的实例。
        这个实例的任何方法都会报错，但只有当真正调用recv、write或者有recv or send事件的时候，才会调用到这些函数，才可能检测到。
        而我们在endpoint对应的函数里spawn_later一个新greenlet而不做join的话，connection的while循环此时已经开始read了。

        之所以不把这个函数实现到connection，是因为shutdown更类似于触发一个close的事件
        用shutdown可以直接触发. how: 0: SHUT_RD, 1: SHUT_WR, else: all
        shutdown后还是会触发close事件以及相关的回调函数，不必担心
        """
        if self.closed():
            return

        try:
            self.ws.stream.handler.socket.shutdown(how)
        except:
            logger.error('shutdown fail.', exc_info=True)

    def set_close_callback(self, callback):
        """
        设置回调
        """
        self._close_callback = callback

    @count_reader
    def read_until(self, terminator, on_read_complete):
        """
        为了兼容调用的方法
        """
        self._read_from_ws(on_read_complete)

    def _read_from_ws(self, on_read_complete):
        """
        接收消息
        """

        if self.closed():
            return

        # 异常也认为服务器断掉链接了，或者该断掉链接
        try:
            data = self.ws.receive()
        except Exception, e:
            logger.exception('rfile.readline fail. e: %s', e)
            data = None

        if not data:
            # 重复调用也没关系
            self.close()

            if self._close_callback:
                self._run_callback(self._close_callback)
            return

        safe_call(on_read_complete, data)

    @count_writer
    def write(self, data, callback=None):
        """
        写数据
        """

        if self.closed():
            return

        safe_call(self.ws.send, data)
        safe_call(callback)

    def closed(self):
        return not self.ws

    def reading(self):
        return bool(self.reader_count)

    def writing(self):
        return bool(self.writer_count)

    def set_nodelay(self, value):
        """
        直接抄的tornado
        """
        pass

    def _run_callback(self, callback, *args):
        """
        替换为kola的safe_call
        """
        return safe_call(callback, *args)
