# -------------------------------------------------------------------------- #
# Copyright 2010-2011, University of Chicago                                      #
#                                                                            #
# Licensed under the Apache License, Version 2.0 (the "License"); you may    #
# not use this file except in compliance with the License. You may obtain    #
# a copy of the License at                                                   #
#                                                                            #
# http://www.apache.org/licenses/LICENSE-2.0                                 #
#                                                                            #
# Unless required by applicable law or agreed to in writing, software        #
# distributed under the License is distributed on an "AS IS" BASIS,          #
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   #
# See the License for the specific language governing permissions and        #
# limitations under the License.                                             #
# -------------------------------------------------------------------------- #

##~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
##
## RECIPE: CA certificate
##
## This recipe installs the CA certificate used to generate the user and
## host certificates in a Globus Provision instance. If no CA certificate
## was specified explicitly in the configuration file, then a self-signed
## certificate will be created on-the-fly by ``gp-start``.
##
## The certificate is installed not just as a trusted certificate, but also
## as a certificate that will be recognized by ``grid-cert-request`` when
## requesting a certificate.
##
## Note that this only installs the certificate, not the private key. To
## set up a node to act as an actual CA (for example, to use with MyProxy)
## use the ``simpleca`` recipe.
##
##~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

require "openssl"

r = cookbook_file "#{node[:scratch_dir]}/gp-ca-cert.pem" do
  source "ca_cert.pem"
  mode 0644
  owner "root"
  group "root"
  action :nothing
end

r.run_action(:create)

node.default[:ca_cert] = OpenSSL::X509::Certificate.new(File.read("#{node[:scratch_dir]}/gp-ca-cert.pem"))
node.default[:ca_cert_hash] = "%08x" % node.default[:ca_cert].subject.hash
  
subject = {}
node.default[:ca_cert].subject.to_a.each do |name,value,n|
  subject[name.downcase]=value
end
node.default[:ca_cert_subject] = subject
# Create the grid-security directory

directory "/etc/grid-security" do
  owner "root"
  group "root"
  mode "0755"
  action :create
end

directory "/etc/grid-security/certificates" do
  owner "root"
  group "root"
  mode "0755"
  action :create
end


# Copy the certificate itself.
# Note that the certificate is generated by demogrid-prepare, and then
# placed in the cookbook's files directory.

cookbook_file "/etc/grid-security/certificates/#{node.default[:ca_cert_hash]}.0" do
  source "ca_cert.pem"
  mode 0644
  owner "root"
  group "root"
end

# Copy a lot of other supporting files.

template "/etc/grid-security/certificates/#{node.default[:ca_cert_hash]}.signing_policy" do
  source "signing_policy.erb"
  mode 0644
  owner "root"
  group "root"
  variables(
    :certificate => node.default[:ca_cert]
  )    
end

template "/etc/grid-security/certificates/globus-user-ssl.conf.#{node.default[:ca_cert_hash]}" do
  source "globus-ssl.conf.erb"
  mode 0644
  owner "root"
  group "root"
  variables(
    :certificate => node.default[:ca_cert],
    :type => :user
  )  
end

template "/etc/grid-security/certificates/globus-host-ssl.conf.#{node.default[:ca_cert_hash]}" do
  source "globus-ssl.conf.erb"
  mode 0644
  owner "root"
  group "root"
  variables(
    :certificate => node.default[:ca_cert],
    :type => :host
  )  
end

template "/etc/grid-security/certificates/grid-security.conf.#{node.default[:ca_cert_hash]}" do
  source "grid-security.conf.erb"
  mode 0644
  owner "root"
  group "root"
  variables(
    :subject => node.default[:ca_cert_subject]
  )    
end

link "/etc/grid-security/globus-host-ssl.conf" do
  to "/etc/grid-security/certificates/globus-host-ssl.conf.#{node.default[:ca_cert_hash]}"
end

link "/etc/grid-security/globus-user-ssl.conf" do
  to "/etc/grid-security/certificates/globus-user-ssl.conf.#{node.default[:ca_cert_hash]}"
end

link "/etc/grid-security/grid-security.conf" do
  to "/etc/grid-security/certificates/grid-security.conf.#{node.default[:ca_cert_hash]}"
end

