# -*- coding: utf-8 -*-
"""gntplib is a pure python implementation of Growl Notification Transport
Protocol client.

Growl Notification Transport Protocol (GNTP) is specified:
http://www.growlforwindows.com/gfw/help/gntp.aspx
"""

import contextlib
import hashlib
import io
import re
import socket


__version__ = '0.1'
__all__ = ['notify', 'Notifier', 'Icon', 'SocketCallback', 'Event']


SUPPORTED_VERSIONS = ['1.0']
MAX_MESSAGE_SIZE = 4096
MAX_LINE_SIZE = 1024
LINE_DELIMITER = '\r\n'
SECTION_DELIMITER = SECTION_BODY_START = SECTION_BODY_END = '\r\n'
MESSAGE_DELIMITER = '\r\n\r\n'
RESPONSE_INFORMATION_LINE_RE = re.compile(
    'GNTP/([^ ]+) (-OK|-ERROR|-CALLBACK) NONE')


def notify(app_name, event_name, title, text='', callback=None):
    """Register notifier and notify event at a time.

    :param app_name: name of the application.
    :param event_name: name of the event.
    :param title: title of the notification.
    :param text: text of the notification.  Defaults to ``''``.
    :param callback: :class:`SocketCallback` instance.  Defaults to `None`.
    """
    notifier = Notifier(app_name, coerce_to_events([event_name]))
    notifier.register()
    notifier.notify(event_name, title, text, callback=callback)


class Notifier(object):
    """Notifier of Growl Notification Transport Protocol (GNTP).

    `event_defs` is list of ``str``, ``unicode``, double or :class:`Event`
    instance.  It is converted to list of :class:`Event` instance as follows:
    ``str`` or ``unicode`` item becomes value of the `name` attribute of
    :class:`Event` instance, whose other attributes are defaults.  Double item
    is expanded to (`name`, `enabled`) tuple, and those values are passed to
    :class:`Event` constructor.  :class:`Event` instance item is used directly.

    :param name: name of the application.
    :param event_defs: definitions of the events.
    :param icon: :class:`Icon` instance or string for the application icon.
                 In Growl 1.3.3, string does not work.  Defaults to `None`.
    """

    def __init__(self, name, event_defs, icon=None):
        self.name = name
        self.icon = icon
        self.events = coerce_to_events(event_defs)

    def register(self):
        """Register this notifier to GNTP server."""
        request = RegisterRequest(self.name, self.icon, self.events)
        with gntp_open(request) as source:
            message = next(generate_messages(source))
            parse_response(message, '-OK')

    def notify(self, name, title, text='', id_=None, sticky=False,
               priority=0, icon=None, coalescing_id=None, callback=None):
        """Notify notification to GNTP server.

        :param name: name of the event.
        :param title: title of the notification.
        :param text: text of the notification.  Defaults to `''`.
        :param id_: unique ID for the notification.  Defaults to `None`.
        :param sticky: if the notification should remain displayed until
                       dismissed by the user.  Defaults to `False`.
        :param priority: display hint for the receiver which may be ignored.
                         A higher number indicates a higher priority.
                         Valid values are between -2 and 2, defaults to `0`.
        :param icon: :class:`Icon` instance or string to display with the
                     notification.  In Growl 1.3.3, string does not work.
                     Defaults to `None`.
        :param coalescing_id: if present, should contain the value of the
                              `id_` of a previously-sent notification.
                              This serves as a hint to the notification system
                              that this notification should replace/update the
                              matching previous notification.  The notification
                              system may ignore this hint.  Defaults to `None`.
        :param callback: :class:`SocketCallback` instance for socket callback
                         or string for url callback. Defaults to `None`.
        """
        notification = Notification(name, title, text, id_=id_, sticky=sticky,
                                    priority=priority, icon=icon,
                                    coalescing_id=coalescing_id,
                                    callback=callback)
        request = NotifyRequest(self.name, notification)
        with gntp_open(request) as source:
            message = next(generate_messages(source))
            parse_response(message, '-OK')

            if notification.callback is not None:
                message = next(generate_messages(source))
                response = parse_response(message, '-CALLBACK')
                notification.callback(response)


class Icon(object):
    """Class for icon.

    :param data: binary data.
    """

    def __init__(self, data):
        self.data = data
        self.unique_id = hashlib.md5(data).hexdigest()
        self.resource = 'x-growl-resource://%s' % self.unique_id

    def get_section(self):
        return ([('Identifier', self.unique_id), ('Length', len(self.data))],
                self.data)


class SocketCallback(object):
    """Abstract base class for socket callback.

    :param context: value of ``Notification-Callback-Context``.
    :param context-type: value of ``Notification-Callback-Context-Type``.
                         Defaults to ``'text/plain; charset=utf-8'``.
    """

    def __init__(self, context, context_type='text/plain; charset=utf-8'):
        self.context = context
        self.context_type = context_type

    def if_clicked(self, response):
        """Template method for clicked callback result.

        :param response: :class:`Response` instance."""

    def if_closed(self, response):
        """Template method for closed callback result.

        :param response: :class:`Response` instance."""

    def if_timedout(self, response):
        """Template method for timedout callback result.

        :param response: :class:`Response` instance."""

    def __call__(self, response):
        """This is the callback.  Delegate to ``if_`` methods depending on
        ``Notification-Callback-Result`` value.

        But TIMEDOUT callback does not occur in my environment.

        :param response: :class:`Response` instance.
        """
        callback_result = response.headers['Notification-Callback-Result']
        delegate_map = {
            'CLICKED': self.if_clicked, 'CLICK': self.if_clicked,
            'CLOSED': self.if_closed, 'CLOSE': self.if_closed,
            'TIMEDOUT': self.if_timedout, 'TIMEOUT': self.if_timedout,
            }
        delegate_map[callback_result](response)

    def get_headers(self):
        """Used to construct ``NOTIFY`` request message."""
        return [('Notification-Callback-Context', self.context),
                ('Notification-Callback-Context-Type', self.context_type)]


class GNTPError(Exception):
    """Base class for GNTP exception."""


class Event(object):
    """Represent notification type."""

    def __init__(self, name, display_name=None, enabled=True, icon=None):
        self.name = name
        self.display_name = display_name
        self.enabled = enabled
        self.icon = coerce_to_icon(icon)

    def get_headers(self):
        results = [('Notification-Name', self.name)]
        if self.display_name is not None:
            results.append(('Notification-Display-Name', self.display_name))
        results.append(('Notification-Enabled', self.enabled))
        if self.icon is not None:
            results.append(('Notification-Icon', self.icon.resource))
        return results

    def get_sections(self):
        results = []
        if isinstance(self.icon, Icon):
            results.append(self.icon.get_section())
        return results


class Notification(object):
    """Represent notification."""

    def __init__(self, name, title, text='', id_=None, sticky=None,
                 priority=None, icon=None, coalescing_id=None, callback=None):
        self.name = name
        self.title = title
        self.text = text
        self.id_ = id_
        self.sticky = sticky
        self.priority = priority
        self.icon = coerce_to_icon(icon)
        self.coalescing_id = coalescing_id
        self.callback = coerce_to_callback(callback)

    def get_headers(self):
        results = [('Notification-Name', self.name)]
        if self.id_ is not None:
            results.append(('Notification-ID', self.id_))
        results.extend([('Notification-Title', self.title),
                        ('Notification-Text', self.text)])
        if self.sticky is not None:
            results.append(('Notification-Sticky', self.sticky))
        if self.priority is not None:
            results.append(('Notification-Priority', self.priority))
        if self.icon is not None:
            results.append(('Notification-Icon', self.icon.resource))
        if self.coalescing_id is not None:
            results.append(('Notification-Coalescing-ID', self.coalescing_id))
        if self.callback is not None:
            results.extend(self.callback.get_headers())
        return results

    def get_sections(self):
        results = []
        if isinstance(self.icon, Icon):
            results.append(self.icon.get_section())
        return results


class BaseRequest(object):
    """Abstract base class for GNTP request."""
    #: Request message type.  Subclasses must override this attribute.
    #: Currently ``REGISTER`` and ``NOTIFY`` are supported.
    message_type = None

    @property
    def message(self):
        """Return utf-8 encoded request message."""
        return 'GNTP/1.0 %s NONE\r\n%s\r\n' % (
            self.message_type, self.message_body)

    @property
    def message_body(self):
        """Return utf-8 encoded message body."""
        writer = io.BytesIO()
        write_headers(writer, self._get_headers())
        write_sections(writer, self._get_sections())
        return writer.getvalue()

    def _get_headers(self):
        """Subclasses must override this method to serialize their headers.

        Return iterable of (`header name`, `header value`) tuple.
        """
        raise NotImplementedError

    def _get_sections(self):
        """Subclasses must override this method to serialize their sections.

        Return iterable of (`section headers`, `section body`) tuple.
        `section headers` is iterable of (`header name`, `header value`)
        tuple.
        """
        raise NotImplementedError


class RegisterRequest(BaseRequest):
    """Represent ``REGISTER`` request.

    :param app_name: name of the application.
    :param app_icon: :class:`Icon` instance or string for the application.
    :param events: list of the events.
    """
    message_type = 'REGISTER'

    def __init__(self, app_name, app_icon, events):
        self.app_name = app_name
        self.app_icon = coerce_to_icon(app_icon)
        self.events = events

    def _get_headers(self):
        results = [('Application-Name', self.app_name)]
        if self.app_icon is not None:
            results.append(('Application-Icon', self.app_icon.resource))
        results.append(('Notifications-Count', len(self.events)))
        return results

    def _get_sections(self):
        results = []
        for event in self.events:
            results.append((event.get_headers(), ''))
        if self.app_icon is not None and isinstance(self.app_icon, Icon):
            results.append(self.app_icon.get_section())
        for event in self.events:
            results.extend(event.get_sections())
        return results


class NotifyRequest(BaseRequest):
    """Represent ``NOTIFY`` request.

    :param app_name: name of the application.
    :param notification: :class:`Notification` instance.
    """
    message_type = 'NOTIFY'

    def __init__(self, app_name, notification):
        self.app_name = app_name
        self.notification = notification

    def _get_headers(self):
        results = [('Application-Name', self.app_name)]
        results.extend(self.notification.get_headers())
        return results

    def _get_sections(self):
        return self.notification.get_sections()


class Response(object):
    """Base class for GNTP response."""

    def __init__(self, message_type, headers):
        self.message_type = message_type
        self.headers = headers


@contextlib.contextmanager
def gntp_open(request, host='localhost', port=23053, timeout=10):
    """Open new GNTP socket and send request message.

    :param request: instance of subclass of :class:`BaseRequest`.
    :param host: host of GNTP server.  Defaults to ``'localhost'``.
    :param port: port of GNTP server.  Defaults to ``23053``.
    :param timeout: timeout.  Defaults to ``10``.
    """
    try:
        sock = socket.create_connection((host, port), timeout=timeout)
        sock.send(request.message)
        yield sock
    finally:
        sock.close()


def generate_messages(sock, size=1024):
    """Generate messages from opened socket."""
    buf = ''
    while True:
        buf += sock.recv(size)
        if not buf:
            break
        # or .find(LINE_DELIMITER + MESSAGE_DELIMITER)?
        pos = buf.find(MESSAGE_DELIMITER)
        if ((pos < 0 and len(buf) >= MAX_MESSAGE_SIZE) or
            (pos > MAX_MESSAGE_SIZE - 4)):
            raise GNTPError('too large message: %r' % buf)
        elif pos > 0:
            pos += 4
            yield buf[:pos]
            buf = buf[pos:]


def parse_response(message, expected_message_type=None):
    """Parse response and return response object."""
    try:
        lines = message.split(LINE_DELIMITER)
        _, message_type = parse_information_line(lines.pop(0))
        if (expected_message_type is not None and
            expected_message_type != message_type):
            raise GNTPError('%s is not expected message type %s' % (
                    message_type, expected_message_type))

        headers = dict([s.strip() for s in line.split(':', 1)]
                       for line in lines if line)
        if message_type == '-ERROR':
            raise GNTPError('%s: %s' % (headers['Error-Code'],
                                        headers['Error-Description']))
        return Response(message_type, headers)
    except (ValueError, GNTPError) as why:
        raise GNTPError(why.args[0], 'original message: %s' % message)


def parse_information_line(line):
    """Parse information line and return tuple (`<version>`,
    `<messagetype>`)."""
    matched = RESPONSE_INFORMATION_LINE_RE.match(line)
    if matched is None:
        raise GNTPError('invalid information line: %r' % line)
    version, message_type = matched.groups()
    if version not in SUPPORTED_VERSIONS:
        raise GNTPError('version %r is not supported' % version)
    return version, message_type


def coerce_to_events(items):
    """Coerce list of event definitions to list of :class:`Event` instances."""
    results = []
    for item in items:
        if isinstance(item, basestring):
            results.append(Event(item, enabled=True))
        elif isinstance(item, tuple):
            name, enabled = item
            results.append(Event(name, enabled=enabled))
        elif isinstance(item, Event):
            results.append(item)
    return results


class _URLIcon(object):

    def __init__(self, url):
        self.resource = url


class _URLCallback(object):

    def __init__(self, url):
        self.url = url

    def get_headers(self):
        return [('Notification-Callback-Target', self.url)]


def coerce_to_icon(obj):
    if obj is None or isinstance(obj, Icon):
        return obj
    if isinstance(obj, basestring):
        return _URLIcon(obj)


def coerce_to_callback(obj):
    if obj is None or isinstance(obj, SocketCallback):
        return obj
    if isinstance(obj, basestring):
        return _URLCallback(obj)


def write_headers(writer, headers):
    """Write utf-8 encoded headers into writer.

    `headers` is iterable of (`header name`, `header value`) tuple.
    If `header value` is instance of unicode, it is encoded.
    Otherwise it is converted with ``str`` function.
    """
    for name, value in headers:
        if isinstance(value, unicode):
            value = value.encode('utf-8')
        elif not isinstance(value, str):
            value = str(value)
        writer.write(name)
        writer.write(': ')
        writer.write(value)
        writer.write(LINE_DELIMITER)


def write_sections(writer, sections):
    """Write utf-8 encoded sections into writer.

    `headers` is iterable of (`section headers`, `section body`) tuple.
    `section headers` is iterable of (`header name`, `header value`)
    tuple.  `section value` is bytes, which is written directly.
    """
    for headers, body in sections:
        writer.write(SECTION_DELIMITER)
        write_headers(writer, headers)
        if body:
            writer.write(SECTION_BODY_START)
            writer.write(body)
            writer.write(SECTION_BODY_END)
