"""This module provides support for asynchronous processing built on Tornado_.

.. _Tornado: http://www.tornadoweb.org/
"""

import logging
import socket

from tornado import gen, httpclient, ioloop, iostream

import gntplib


__all__ = ['AsyncNotifier', 'AsyncIcon']


logger = logging.getLogger(__name__)


class AsyncNotifier(gntplib.Notifier):
    """Asynchronous Notifier of Growl Notification Transport Protocol (GNTP).

    `event_defs` is list of ``str``, ``unicode``, double or :class:`Event`
    instance.  It is converted to list of :class:`Event` instance as follows:
    ``str`` or ``unicode`` item becomes value of the `name` attribute of
    :class:`Event` instance, whose other attributes are defaults.  Double item
    is expanded to (`name`, `enabled`) tuple, and those values are passed to
    :class:`Event` constructor.  :class:`Event` instance item is used directly.

    :param name: the name of the application.
    :param event_defs: the definitions of the events.
    :param icon: :class:`Icon` instance or string for the application icon.
                 Defaults to `None`.
    :param io_loop: :class:`tornado.ioloop.IOLoop` instance.

    .. note:: In Growl 1.3.3, `icon` of string does not work.
    """

    def __init__(self, name, event_defs, icon=None, io_loop=None, **kwargs):
        io_loop = io_loop or ioloop.IOLoop.instance()
        gntplib.Notifier.__init__(self, name, event_defs, icon,
                                  AsyncGNTPClient, io_loop=io_loop, **kwargs)


class AsyncGNTPConnection(gntplib.GNTPConnectionCallbacksMixin):
    """Asynchronous GNTP connection built on Tornado."""

    def __init__(self, address, timeout, final_callback,
                 callback_message_callback=None, socket_callback=None,
                 io_loop=None):
        sock = socket.create_connection(address, timeout=timeout)
        self.stream = iostream.IOStream(sock, io_loop=io_loop)
        self.final_callback = final_callback
        self.callback_message_callback = (callback_message_callback or
                                          self.on_callback_message)
        self.socket_callback = socket_callback
        self.io_loop = io_loop

    @gen.engine
    def write_message(self, request):
        """Send the request to the GNTP server.  If the request contains
        :class:`AsyncIcon` instances, after all their fetches are completed,
        the request will be sent the GNTP server."""
        async_icons = collect_async_icons(request)
        if async_icons:
            yield gen.Task(fetch_async_icons_in_parallel,
                           async_icons,
                           self.io_loop)
        self.stream.write(request.message)

    def read_message(self, callback):
        """Read a message from opened stream and run callback with it."""
        delimiter = gntplib.LINE_DELIMITER + gntplib.MESSAGE_DELIMITER
        self.stream.read_until(delimiter, callback)

    def close(self):
        """Close the connection."""
        self.stream.close()
        self.stream = None


class AsyncGNTPClient(gntplib.GNTPClient):
    """Asynchronous GNTP client built on Tornado.

    :param host: host of GNTP server.  Defaults to ``'localhost'``.
    :param port: port of GNTP server.  Defaults to ``23053``.
    :param timeout: timeout.  Defaults to ``10``.
    :param io_loop: :class:`tornado.ioloop.IOLoop` instance.
    """

    connection_class = AsyncGNTPConnection

    def __init__(self, host='localhost', port=23053, timeout=10, io_loop=None):
        gntplib.GNTPClient.__init__(self, host, port, timeout)
        self.io_loop = io_loop

    def _connect(self, final_callback, **kwargs):
        """Connect to GNTP server and return the connection."""
        return gntplib.GNTPClient._connect(self, final_callback,
                                           io_loop=self.io_loop, **kwargs)


class AsyncIcon(gntplib.RawIcon):

    def __init__(self, url):
        gntplib.RawIcon.__init__(self, None)
        self.url = url


@gen.engine
def fetch_async_icons_in_parallel(async_icons, io_loop, callback):
    """Fetch :class:`AsyncIcon`'s url in parallel."""
    http_client = httpclient.AsyncHTTPClient(io_loop=io_loop)
    responses = yield [gen.Task(http_client.fetch, icon.url)
                       for icon in async_icons]
    for icon, response in zip(async_icons, responses):
        if response.error:
            logger.warning('failed to fetch %r: %s', icon.url, response.error)
            icon.data = None
        else:
            icon.data = response.body
    callback(async_icons)


def collect_async_icons(request):
    """Collect :class:`AsyncIcon` instances from given request."""
    if isinstance(request, gntplib.RegisterRequest):
        icons = [request.app_icon] + [e.icon for e in request.events]
    elif isinstance(request, gntplib.NotifyRequest):
        icons = [request.notification.icon]
    else:
        # not reached
        raise ValueError('unrecognized request: %r' % request)
    return list(set(i for i in icons if isinstance(i, AsyncIcon)))
