"""This module provides support for asynchronous processing built on Tornado_.

.. _Tornado: http://www.tornadoweb.org/
"""

from __future__ import unicode_literals
import logging
import socket

from tornado import gen, httpclient, ioloop, iostream

import gntplib


__all__ = ['AsyncNotifier', 'AsyncIcon']


logger = logging.getLogger(__name__)


class AsyncNotifier(gntplib.Notifier):
    """Asynchronous Notifier of Growl Notification Transport Protocol (GNTP).

    Same as :class:`~gntplib.Notifier` except the following:

    * uses :class:`AsyncGNTPClient` as `gntp_client_class`.
    * accepts :class:`tornado.ioloop.IOLoop` instance by additional
      `io_loop` keyword argument.

    :param name: the name of the application.
    :param event_defs: the definitions of the notifications.
    :param icon: url string or an instance of subclass of :class:`BaseIcon`
                 for the icon of the application.  Defaults to `None`.
    :param io_loop: :class:`tornado.ioloop.IOLoop` instance.
    """

    def __init__(self, name, event_defs, icon=None, io_loop=None, **kwargs):
        io_loop = io_loop or ioloop.IOLoop.instance()
        gntplib.Notifier.__init__(self, name, event_defs, icon,
                                  AsyncGNTPClient, io_loop=io_loop, **kwargs)


class AsyncGNTPConnection(gntplib.GNTPConnectionCallbacksMixin):
    """Asynchronous GNTP connection built on Tornado."""

    def __init__(self, address, timeout, final_callback, socket_callback=None,
                 io_loop=None):
        sock = socket.create_connection(address, timeout=timeout)
        self.stream = iostream.IOStream(sock, io_loop=io_loop)
        self.final_callback = final_callback
        self.socket_callback = socket_callback
        self.io_loop = io_loop

    @gen.engine
    def write_message(self, request):
        """Send the request to the GNTP server.  If the request contains
        :class:`AsyncIcon` instances, the request will be sent the GNTP server
        after all their fetches are completed asynchronously."""
        async_icons = collect_async_icons(request)
        if async_icons:
            yield gen.Task(fetch_async_icons_in_parallel,
                           async_icons,
                           self.io_loop)
        self.stream.write(request.message)

    def read_message(self, callback):
        """Read a message from opened stream and run callback with it."""
        delimiter = gntplib.LINE_DELIMITER + gntplib.MESSAGE_DELIMITER
        self.stream.read_until(delimiter, callback)

    def close(self):
        """Close the stream."""
        self.stream.close()
        self.stream = None


class AsyncGNTPClient(gntplib.GNTPClient):
    """Asynchronous GNTP client built on Tornado.

    :param host: host of GNTP server.  Defaults to ``'localhost'``.
    :param port: port of GNTP server.  Defaults to ``23053``.
    :param timeout: timeout.  Defaults to ``10``.
    :param io_loop: :class:`tornado.ioloop.IOLoop` instance.
    """

    connection_class = AsyncGNTPConnection

    def __init__(self, host='localhost', port=23053, timeout=10, io_loop=None):
        gntplib.GNTPClient.__init__(self, host, port, timeout)
        self.io_loop = io_loop

    def _connect(self, final_callback, **kwargs):
        """Connect to the GNTP server and return the connection."""
        return gntplib.GNTPClient._connect(self, final_callback,
                                           io_loop=self.io_loop, **kwargs)


class AsyncIcon(gntplib.RawIcon):
    """Class for asynchronous icon.

    :param url: url string of icon image.
    """

    def __init__(self, url):
        gntplib.RawIcon.__init__(self, None)
        self.url = url


@gen.engine
def fetch_async_icons_in_parallel(async_icons, io_loop, callback):
    """Fetch :class:`AsyncIcon`'s url in parallel."""
    http_client = httpclient.AsyncHTTPClient(io_loop=io_loop)
    responses = yield [gen.Task(http_client.fetch, icon.url)
                       for icon in async_icons]
    for icon, response in zip(async_icons, responses):
        if response.error:
            logger.warning('failed to fetch %r: %s', icon.url, response.error)
            icon.data = None
        else:
            icon.data = response.body
    callback(async_icons)


def collect_async_icons(request):
    """Collect :class:`AsyncIcon` instances from given request."""
    if isinstance(request, gntplib.RegisterRequest):
        icons = [request.app_icon] + [e.icon for e in request.events]
    elif isinstance(request, gntplib.NotifyRequest):
        icons = [request.notification.icon]
    else:
        # not reached
        icons = []
    return list(set(i for i in icons if isinstance(i, AsyncIcon)))
