"""This module provides core functionalities of gntplib.

gntplib is a Growl Notification Transport Protocol (GNTP_) client library in
Python.

.. _GNTP: http://www.growlforwindows.com/gfw/help/gntp.aspx
"""

from __future__ import unicode_literals
import hashlib
import io
import re
import socket

from . import keys
from .compat import text_type


__version__ = '0.4'
__all__ = ['notify', 'Notifier', 'RawIcon', 'SocketCallback', 'Event']


SUPPORTED_VERSIONS = ['1.0']
MAX_MESSAGE_SIZE = 4096
MAX_LINE_SIZE = 1024
LINE_DELIMITER = b'\r\n'
SECTION_DELIMITER = SECTION_BODY_START = SECTION_BODY_END = b'\r\n'
MESSAGE_DELIMITER = b'\r\n\r\n'
RESPONSE_INFORMATION_LINE_RE = re.compile(
    b'GNTP/([^ ]+) (-OK|-ERROR|-CALLBACK) NONE')


def notify(app_name, event_name, title, text=''):
    """Register a notifier and send a notification at a time.

    :param app_name: the name of the application.
    :param event_name: the name of the notification.
    :param title: the title of the notification.
    :param text: the text of the notification.  Defaults to ``''``.
    """
    notifier = Notifier(app_name, coerce_to_events([event_name]))
    notifier.register()
    notifier.notify(event_name, title, text)


class Notifier(object):
    """Notifier of Growl Notification Transport Protocol (GNTP).

    Currently this class supports ``REGISTER`` and ``NOTIFY`` requests.  They
    are sent by :meth:`register()` and :meth:`notify()` methods respectively.
    These methods can accept the optional final callback as `callback` keyword
    argument, which run after closing the connection with the GNTP server.

    `event_defs` is a list of ``str``, ``unicode``, double (of ``str`` and
    ``bool``) or :class:`Event` instance.  It is converted to a list of
    :class:`Event` instance as follows:
    ``str`` or ``unicode`` item becomes value of the `name` attribute of
    :class:`Event` instance, whose other attributes are defaults.  Double item
    is expanded to (`name`, `enabled`) tuple, and those values are passed to
    :class:`Event` constructor.  :class:`Event` instance item is used directly.

    Optional keyword arguments are passed to the `gntp_client_class`
    constructor.

    :param name: the name of the application.
    :param event_defs: the definitions of the notifications.
    :param icon: url string or an instance of subclass of :class:`BaseIcon`
                 for the icon of the application.  Defaults to `None`.
    :param gntp_client_class: GNTP client class.  If it is `None`,
                              :class:`GNTPClient` is used.  Defaults to `None`.

    .. note:: In Growl 1.3.3, `icon` of url string does not work.
    """

    def __init__(self, name, event_defs, icon=None, gntp_client_class=None,
                 **kwargs):
        self.name = name
        self.icon = icon
        self.events = coerce_to_events(event_defs)
        if not self.events:
            raise GNTPError('You have to set at least one notification type')
        if gntp_client_class is None:
            gntp_client_class = GNTPClient
        self.gntp_client = gntp_client_class(**kwargs)

    def register(self, callback=None):
        """Register this notifier to the GNTP server.

        :param callback: the callback run after closing the connection with
                         the GNTP server.  Defaults to `None`.
        """
        request = RegisterRequest(self.name, self.icon, self.events)
        self.gntp_client.process_request(request, callback)

    def notify(self, name, title, text='', id_=None, sticky=False,
               priority=0, icon=None, coalescing_id=None, callback=None,
               gntp_callback=None, **socket_callback_options):
        """Send a notification to the GNTP server.

        :param name: the name of the notification.
        :param title: the title of the notification.
        :param text: the text of the notification.  Defaults to `''`.
        :param id_: the unique ID for the notification.  If set, this should be
                    unique for every request.  Defaults to `None`.
        :param sticky: if set to `True`, the notification remains displayed
                       until dismissed by the user.  Defaults to `False`.
        :param priority: the display hint for the receiver which may be
                         ignored.  A higher number indicates a higher priority.
                         Valid values are between -2 and 2, defaults to `0`.
        :param icon: url string or an instance of subclass of :class:`BaseIcon`
                     to display with the notification.  Defaults to `None`.
        :param coalescing_id: if set, should contain the value of the `id_` of
                              a previously-sent notification.
                              This serves as a hint to the notification system
                              that this notification should replace/update the
                              matching previous notification.  The notification
                              system may ignore this hint.  Defaults to `None`.
        :param callback: the callback run after closing the connection with
                         the GNTP server.  Defaults to `None`.
        :param gntp_callback: url string for url callback or
                              :class:`SocketCallback` instance for socket
                              callback.  Defaults to `None`.
        :param socket_callback_options: the keyword arguments to be used to
                                        instantiating :class:`SocketCallback`
                                        for socket callback.  About acceptable
                                        keyword arguments,
                                        see :class:`SocketCallback`.

        .. note:: In Growl 1.3.3, `icon` of url string does not work.

        .. note:: Growl for Windows v2.0+ and Growl v1.3+ require
                  `coalescing_id` to be the same on both the original and
                  updated notifcation, ignoring the value of `id_`.
        """
        notification = Notification(name, title, text, id_=id_, sticky=sticky,
                                    priority=priority, icon=icon,
                                    coalescing_id=coalescing_id,
                                    gntp_callback=gntp_callback,
                                    **socket_callback_options)
        request = NotifyRequest(self.name, notification)
        self.gntp_client.process_request(
            request, callback, socket_callback=notification.socket_callback)


class GNTPError(Exception):
    """Base class for GNTP exception."""


class Event(object):
    """Represent notification type.

    :param name: the name of the notification.
    :param display_name: the display name of the notification, which is
                         appeared at the Applications tab of the Growl
                         Preferences.  Defaults to `None`.
    :param enabled: indicates if the notification should be enabled by
                    default.  Defaults to `True`.
    :param icon: url string or an instance of subclass of :class:`BaseIcon`
                 for the default icon to display with the notifications of this
                 notification type.  Defaults to `None`.

    .. note:: In Growl 1.3.3, `icon` does not work.
    """

    def __init__(self, name, display_name=None, enabled=True, icon=None):
        self.name = name
        self.display_name = display_name
        self.enabled = enabled
        self.icon = coerce_to_icon(icon)


class Notification(object):
    """Represent notification."""

    def __init__(self, name, title, text='', id_=None, sticky=None,
                 priority=None, icon=None, coalescing_id=None,
                 gntp_callback=None, **socket_callback_options):
        self.name = name
        self.title = title
        self.text = text
        self.id_ = id_
        self.sticky = sticky
        self.priority = priority
        self.icon = coerce_to_icon(icon)
        self.coalescing_id = coalescing_id
        self.callback = coerce_to_callback(gntp_callback,
                                           **socket_callback_options)

    @property
    def socket_callback(self):
        if isinstance(self.callback, SocketCallback):
            return self.callback


class RegisterRequest(object):
    """Represent ``REGISTER`` request.

    :param app_name: the name of the application.
    :param app_icon: url string or an instance of subclass of :class:`BaseIcon`
                     for the icon of the application.
    :param events: the list of :class:`Event` instances.

    .. note:: In Growl 1.3.3, `app_icon` of url string does not work.
    """
    message_type = 'REGISTER'

    def __init__(self, app_name, app_icon, events):
        self.app_name = app_name
        self.app_icon = coerce_to_icon(app_icon)
        self.events = events

    def write_into(self, writer):
        writer.write_register_request(self)


class NotifyRequest(object):
    """Represent ``NOTIFY`` request.

    :param app_name: the name of the application.
    :param notification: :class:`Notification` instance.
    """
    message_type = 'NOTIFY'

    def __init__(self, app_name, notification):
        self.app_name = app_name
        self.notification = notification

    def write_into(self, writer):
        writer.write_notify_request(self)


class Response(object):
    """Base class for GNTP response.

    :param message_type: <messagetype> of the response.  `'-OK'`, `'-ERROR'` or
                         `'-CALLBACK'`.
    :param headers: headers of the response.
    """

    def __init__(self, message_type, headers):
        self.message_type = message_type
        self.headers = headers


class BaseGNTPConnection(object):
    """Abstract base class for GNTP connection."""

    def __init__(self, final_callback, socket_callback=None):
        self.final_callback = final_callback
        self.socket_callback = socket_callback

    def on_ok_message(self, message):
        r"""Callback for ``-OK`` response.

        :param message: string of response terminated by `'\\r\\n\\r\\n'`.
        """
        try:
            response = parse_response(message, '-OK')
            if self.socket_callback is not None:
                self.read_message(self.on_callback_message)
        finally:
            if self.socket_callback is None:
                self.close()
        if self.socket_callback is None and self.final_callback is not None:
            self.final_callback(response)

    def on_callback_message(self, message):
        r"""Callback for ``-CALLBACK`` response.

        :param message: string of response terminated by `'\\r\\n\\r\\n'`.
        """
        try:
            response = parse_response(message, '-CALLBACK')
            callback_result = self.socket_callback(response)
        finally:
            self.close()
        if self.final_callback is not None:
            self.final_callback(callback_result)

    def write_message(self, message):
        """Subclasses must override this method to send a message to the GNTP
        server."""
        raise NotImplementedError

    def read_message(self, callback):
        """Subclasses must override this method to receive a message from the
        GNTP server."""
        raise NotImplementedError

    def close(self):
        """Subclasses must override this method to close the connection with
        the GNTP server."""
        raise NotImplementedError


class GNTPConnection(BaseGNTPConnection):
    """Represent the connection with the GNTP server."""

    def __init__(self, address, timeout, final_callback, socket_callback=None):
        BaseGNTPConnection.__init__(self, final_callback, socket_callback)
        self.sock = socket.create_connection(address, timeout=timeout)

    def write_message(self, message):
        """Send the request message to the GNTP server."""
        self.sock.send(message)

    def read_message(self, callback):
        """Read a message from opened socket and run callback with it."""
        message = next(generate_messages(self.sock))
        callback(message)

    def close(self):
        """Close the socket."""
        self.sock.close()
        self.sock = None


class GNTPClient(object):
    """GNTP client.

    :param host: host of GNTP server.  Defaults to `'localhost'`.
    :param port: port of GNTP server.  Defaults to `23053`.
    :param timeout: timeout in seconds.  Defaults to `10`.
    :param password: the password used in creating the key.
    :param key_hashing: the type of hash algorithm used in creating the key.
                        It is `keys.MD5`, `keys.SHA1`, `keys.SHA256` or
                        `keys.SHA512`.  Defaults to `keys.SHA256`.
    :param encryption: the tyep of encryption algorithm used.
                       It is `None`, `ciphers.AES`, `ciphers.DES` or
                       `ciphers.3DES`.  `None` means no encryption.
                       Defaults to `None`.
    :param connection_class: GNTP connection class.  If it is `None`,
                             :class:`GNTPConnection` is used.  Defaults to
                             `None`.
    """

    def __init__(self, host='localhost', port=23053, timeout=10,
                 password=None, key_hashing=keys.SHA256, encryption=None,
                 connection_class=None):
        self.address = (host, port)
        self.timeout = timeout
        self.connection_class = connection_class or GNTPConnection
        if (encryption is not None and
            encryption.key_size > key_hashing.key_size):
            raise GNTPError('key_hashing key size (%s:%d) must be at'
                            ' least encryption key size (%s:%d)' % (
                    key_hashing.algorithm_id, key_hashing.key_size,
                    encryption.algorithm_id, encryption.key_size))
        self.packer_factory = RequestPackerFactory(password, key_hashing,
                                                   encryption)

    def process_request(self, request, callback, **kwargs):
        """Process a request.

        :param callback: the final callback run after closing connection.
        """
        packer = self.packer_factory.create()
        message = packer.pack(request)
        conn = self._connect(callback, **kwargs)
        conn.write_message(message)
        conn.read_message(conn.on_ok_message)

    def _connect(self, final_callback, **kwargs):
        """Connect to the GNTP server and return the connection."""
        return self.connection_class(self.address, self.timeout,
                                     final_callback, **kwargs)


def generate_messages(sock, size=1024):
    """Generate messages from opened socket."""
    buf = b''
    while True:
        buf += sock.recv(size)
        if not buf:
            break
        pos = buf.find(MESSAGE_DELIMITER)
        if ((pos < 0 and len(buf) >= MAX_MESSAGE_SIZE) or
            (pos > MAX_MESSAGE_SIZE - 4)):
            raise GNTPError('too large message: %r' % buf)
        elif pos > 0:
            pos += 4
            yield buf[:pos]
            buf = buf[pos:]


def parse_response(message, expected_message_type=None):
    """Parse response and return response object."""
    try:
        lines = [line for line in message.split(LINE_DELIMITER) if line]
        _, message_type = parse_information_line(lines.pop(0))
        if (expected_message_type is not None and
            expected_message_type != message_type):
            raise GNTPError('%s is not expected message type %s' % (
                    message_type, expected_message_type))

        headers = dict([s.strip().decode('utf-8') for s in line.split(b':', 1)]
                       for line in lines)
        if message_type == '-ERROR':
            raise GNTPError('%s: %s' % (headers['Error-Code'],
                                        headers['Error-Description']))
        return Response(message_type, headers)
    except ValueError as exc:
        raise GNTPError(exc.args[0], 'original message: %r' % message)
    except GNTPError as exc:
        exc.args = (exc.args[0], 'original message: %r' % message)
        raise exc


def parse_information_line(line):
    """Parse information line and return tuple (`<version>`,
    `<messagetype>`)."""
    matched = RESPONSE_INFORMATION_LINE_RE.match(line)
    if matched is None:
        raise GNTPError('invalid information line: %r' % line)
    version, message_type = [s.decode('utf-8') for s in matched.groups()]
    if version not in SUPPORTED_VERSIONS:
        raise GNTPError("version '%s' is not supported" % version)
    return version, message_type


def coerce_to_events(items):
    """Coerce the list of the event definitions to the list of :class:`Event`
    instances."""
    results = []
    for item in items:
        if isinstance(item, (bytes, text_type)):
            results.append(Event(item, enabled=True))
        elif isinstance(item, tuple):
            name, enabled = item
            results.append(Event(name, enabled=enabled))
        elif isinstance(item, Event):
            results.append(item)
    return results


class BaseIcon(object):
    """Abstract base class for icon."""

    def __init__(self, data):
        self.data = data

    def resource(self):
        """Used to write header of this instance into the writer."""
        raise NotImplementedError

    def write_into(self, writer):
        """Used to write section of this instance into the writer."""
        raise NotImplementedError


class RawIcon(BaseIcon):
    """Class for icon of <uniqueid> data type.

    :param data: bytes of icon image.
    """

    def __init__(self, data):
        BaseIcon.__init__(self, data)
        self._unique_id = None

    def unique_id(self):
        """Return the value of <uniqueid> data type."""
        if self.data is not None and self._unique_id is None:
            self._unique_id = hashlib.md5(self.data).hexdigest()
        return self._unique_id

    def resource(self):
        """Return the resource string used in request headers."""
        if self.data is not None:
            return 'x-growl-resource://%s' % self.unique_id()

    def write_into(self, writer):
        if self.data is not None:
            writer.write_raw_icon(self)


class URLIcon(BaseIcon):
    """Class for icon of <url> data type."""

    def __init__(self, url):
        BaseIcon.__init__(self, url)

    def resource(self):
        return self.data

    def write_into(self, writer):
        pass


def coerce_to_icon(obj):
    if obj is None or isinstance(obj, BaseIcon):
        return obj
    if isinstance(obj, (bytes, text_type)):
        return URLIcon(obj)


class SocketCallback(object):
    """Base class for socket callback.

    Each of the callbacks takes one positional argument, which is
    :class:`Response` instance.

    :param context: value of ``Notification-Callback-Context``.
                    Defaults to ``'None'``.
    :param context-type: value of ``Notification-Callback-Context-Type``.
                         Defaults to ``'None'``.
    :param on_click: the callback run at ``CLICKED`` callback result.
    :param on_close: the callback run at ``CLOSED`` callback result.
    :param on_timeout: the callback run at ``TIMEDOUT`` callback result.

    .. note:: TIMEDOUT callback does not occur in my Growl 1.3.3.
    """

    def __init__(self, context='None', context_type='None',
                 on_click=None, on_close=None, on_timeout=None):
        self.context = context
        self.context_type = context_type
        self.on_click_callback = on_click
        self.on_close_callback = on_close
        self.on_timeout_callback = on_timeout

    def on_click(self, response):
        """Run ``CLICKED`` event callback."""
        if self.on_click_callback is not None:
            return self.on_click_callback(response)

    def on_close(self, response):
        """Run ``CLOSED`` event callback."""
        if self.on_close_callback is not None:
            return self.on_close_callback(response)

    def on_timeout(self, response):
        """Run ``TIMEDOUT`` event callback."""
        if self.on_timeout_callback is not None:
            return self.on_timeout_callback(response)

    def __call__(self, response):
        """This is the callback.  Delegate to ``on_`` methods depending on
        ``Notification-Callback-Result`` value.

        :param response: :class:`Response` instance.
        """
        callback_result = response.headers['Notification-Callback-Result']
        delegate_map = {
            'CLICKED': self.on_click, 'CLICK': self.on_click,
            'CLOSED': self.on_close, 'CLOSE': self.on_close,
            'TIMEDOUT': self.on_timeout, 'TIMEOUT': self.on_timeout,
            }
        return delegate_map[callback_result](response)

    def write_into(self, writer):
        writer.write_socket_callback(self)


class URLCallback(object):
    """Class for url callback."""

    def __init__(self, url):
        self.url = url

    def write_into(self, writer):
        writer.write_url_callback(self)


def coerce_to_callback(gntp_callback=None, **socket_callback_options):
    """Return :class:`URLCallback` instance for url callback or
    :class:`SocketCallback` instance for socket callback.

    If `gntp_callback` is not `None`, `socket_callback_options` must be empty.
    Moreover, if `gntp_callback` is string, then a instance of
    :class:`URLCallback` is returned.  Otherwise, `gntp_callback` is returned
    directly.

    If `gntp_callback` is `None` and `socket_callback_options` is not empty,
    new instance of :class:`SocketCallback` is created from given keyword
    arguments and it is returned.  Acceptable keyword arguments are same as
    constructor's of :class:`SocketCallback`.
    """
    if gntp_callback is not None:
        if socket_callback_options:
            raise GNTPError('If gntp_callback is not None,'
                            ' socket_callback_options must be empty')
        if isinstance(gntp_callback, (bytes, text_type)):
            return URLCallback(gntp_callback)
        else:
            return gntp_callback
    if socket_callback_options:
        return SocketCallback(**socket_callback_options)


class _NullCipher(object):
    """Null object for the encryption of messages."""

    algorithm = None
    algorithm_id = 'NONE'
    encrypt = lambda self, text: text
    decrypt = lambda self, text: text
    __bool__ = lambda self: False
    __nonzero__ = __bool__


NullCipher = _NullCipher()


class RequestPackerFactory(object):
    """The factory of :class:`RequestPacker`.

    If `password` is None, `hashing` and `encryption` are ignored.
    """

    def __init__(self, password=None, hashing=keys.SHA256, encryption=None):
        self.password = password
        self.hashing = password and hashing
        self.encryption = (password and encryption) or NullCipher

    def create(self):
        """Create an instance of :class:`RequestPacker` and return it."""
        key = self.password and self.hashing.key(self.password)
        cipher = self.encryption and self.encryption.cipher(key)
        return RequestPacker(key, cipher)


class RequestPacker(object):
    """The serializer of requests.

    `key` and `cipher` have random-generated salt and iv respectively.

    :param key: an instance of :class:`keys.Key`.
    :param cipher: an instance of :class:`ciphers.Cipher` or `NullCipher`.
    """

    def __init__(self, key=None, cipher=None):
        self.key = key
        self.cipher = cipher or NullCipher

    def pack(self, request):
        """Return utf-8 encoded request message."""
        return (self._information_line(request) +
                LINE_DELIMITER +
                self._headers(request) +
                self._sections(request) +
                LINE_DELIMITER)

    def _information_line(self, request):
        """Return utf-8 encoded information line."""
        result = (b'GNTP/1.0 ' +
                  request.message_type.encode('utf-8') +
                  b' ' +
                  self.cipher.algorithm_id.encode('utf-8'))
        if self.cipher.algorithm is not None:
            result += b':' + self.cipher.iv_hex
        if self.key is not None:
            result += (b' ' +
                       self.key.algorithm_id.encode('utf-8') +
                       b':' +
                       self.key.key_hash_hex +
                       b'.' +
                       self.key.salt_hex)
        return result

    def _headers(self, request):
        """Return utf-8 encoded headers."""
        writer = io.BytesIO()
        header_writer = HeaderWriter(writer)
        request.write_into(header_writer)
        headers = writer.getvalue()
        result = self.cipher.encrypt(headers)
        if self.cipher.algorithm is not None:
            result += LINE_DELIMITER
        return result

    def _sections(self, request):
        """Return utf-8 encoded message body."""
        writer = io.BytesIO()
        header_writer = HeaderWriter(writer)
        section_writer = SectionWriter(writer, header_writer, self.cipher)
        request.write_into(section_writer)
        return writer.getvalue()


class HeaderWriter(object):

    def __init__(self, writer):
        self.writer = writer

    def write_register_request(self, request):
        self.write(b'Application-Name', request.app_name)
        self.write_icon(request.app_icon, b'Application-Icon')
        self.write(b'Notifications-Count', len(request.events))
        for event in request.events:
            self.writer.write(LINE_DELIMITER)
            self.write_event(event)

    def write_notify_request(self, request):
        self.write(b'Application-Name', request.app_name)
        self.write_notification(request.notification)

    def write_event(self, event):
        self.write(b'Notification-Name', event.name)
        self.write(b'Notification-Display-Name', event.display_name)
        self.write(b'Notification-Enabled', event.enabled)
        self.write_icon(event.icon, b'Notification-Icon')

    def write_notification(self, notification):
        self.write(b'Notification-Name', notification.name)
        self.write(b'Notification-ID', notification.id_)
        self.write(b'Notification-Title', notification.title)
        self.write(b'Notification-Text', notification.text)
        self.write(b'Notification-Sticky', notification.sticky)
        self.write(b'Notification-Priority', notification.priority)
        self.write_icon(notification.icon, b'Notification-Icon')
        self.write(b'Notification-Coalescing-ID', notification.coalescing_id)
        if notification.callback is not None:
            notification.callback.write_into(self)

    def write_socket_callback(self, callback):
        self.write(b'Notification-Callback-Context', callback.context)
        self.write(b'Notification-Callback-Context-Type',
                   callback.context_type)

    def write_url_callback(self, callback):
        self.write(b'Notification-Callback-Target', callback.url)

    def write_icon(self, icon, name):
        if icon is not None:
            self.write(name, icon.resource())

    def write(self, name, value):
        """Write utf-8 encoded headers into writer.

        :param name: the name of the header.
        :param value: the value of the header.
        """
        if value is not None:
            if not isinstance(value, bytes):
                value = text_type(value).encode('utf-8')
            self.writer.write(name)
            self.writer.write(b': ')
            self.writer.write(value)
            self.writer.write(LINE_DELIMITER)


class SectionWriter(object):

    def __init__(self, writer, header_writer, cipher):
        self.writer = writer
        self.header_writer = header_writer
        self.cipher = cipher

    def write_register_request(self, request):
        self.write_icon(request.app_icon)
        for event in request.events:
            self.write_event(event)

    def write_notify_request(self, request):
        self.write_notification(request.notification)

    def write_event(self, event):
        self.write_icon(event.icon)

    def write_notification(self, notification):
        self.write_icon(notification.icon)

    def write_raw_icon(self, icon):
        icon_data = self.cipher.encrypt(icon.data)
        headers = [(b'Identifier', icon.unique_id()),
                   (b'Length', len(icon_data))]
        self.write(headers, icon_data)

    def write_icon(self, icon):
        if icon is not None:
            icon.write_into(self)

    def write(self, headers, body):
        """Write utf-8 encoded sections into writer.

        :param headers: the iterable of (`name`, `value`) tuple of the header.
        :param body: bytes of section body.
        """
        self.writer.write(SECTION_DELIMITER)
        for name, value in headers:
            self.header_writer.write(name, value)
        self.writer.write(SECTION_BODY_START)
        self.writer.write(body)
        self.writer.write(SECTION_BODY_END)
