#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Cookie vulnerabilities.
"""

__license__= """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2013

Authors:
  Daniel Garcia Garcia a.k.a cr0hn | cr0hn<@>cr0hn.com
  Mario Vilas | mvilas<@>gmail.com

Golismero project site: https://github.com/golismero
Golismero project mail: golismero.project<@>gmail.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

__all__ = ["CookieVulnerability"]

from .. import Vulnerability
from ... import identity
from ...information.http import HTTP_Response


#------------------------------------------------------------------------------
class CookieVulnerability(Vulnerability):
    """
    Cookie Vulnerability.

    An unspecified vulnerability has been found in an HTTP cookie.
    """


    #--------------------------------------------------------------------------
    def __init__(self, response, cookie, **kwargs):
        """
        :param response: HTTP response where the vulnerable cookie was found.
        :type response: HTTP_Response

        :param cookie: Name of the vulnerable cookie.
        :type cookie: str

        """

        # Validate the data types.
        if not isinstance(response, HTTP_Response):
            raise TypeError("Expected HTTP_Response, got %r instead"
                            % type(response))
        if type(cookie) is not str:
            raise TypeError("Expected str, got %r instead" % type(cookie))

        # Save the HTTP response ID.
        self.__response_id = response.identity

        # Save the cookie name.
        self.__cookie = cookie

        # Parent constructor.
        super(CookieVulnerability, self).__init__(**kwargs)

        # Link the vulnerability to the response.
        self.add_information(response)

    __init__.__doc__ += Vulnerability.__init__.__doc__


    #--------------------------------------------------------------------------
    @identity
    def response_id(self):
        """
        :returns: Identity hash of the HTTPResponse
            where the vulnerable cookie was found.
        :rtype: str
        """
        return self.__response_id


    #--------------------------------------------------------------------------
    @property
    def response(self):
        """
        :returns: HTTP response where the vulnerable cookie was found.
        :rtype: HTTP_Response
        """
        return self.resolve(self.response_id)


    #--------------------------------------------------------------------------
    @identity
    def cookie(self):
        """
        :returns: Name of the vulnerable cookie.
        :rtype: str
        """
        return self.__cookie
