#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__= """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2013

Authors:
  Daniel Garcia Garcia a.k.a cr0hn | cr0hn<@>cr0hn.com
  Mario Vilas | mvilas<@>gmail.com

Golismero project site: https://github.com/golismero
Golismero project mail: golismero.project<@>gmail.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

__all__ = ["OutdatedSoftware"]

from .. import Vulnerability
from ... import identity
from ...resource import Resource


#------------------------------------------------------------------------------
class OutdatedSoftware(Vulnerability):
    """
    Outdated Software.

    Outdated, potentially vulnerable software was found.

    Apply all missing patches or upgrade to a newer version.
    """

    DEFAULTS = Vulnerability.DEFAULTS.copy()
    DEFAULTS["level"] = "high"
    DEFAULTS["cvss_base"] = "8.5"
    DEFAULTS["references"] = (
        "https://www.owasp.org/index.php/Top_10_2013-A5-Security_Misconfiguration",
    )


    #--------------------------------------------------------------------------
    def __init__(self, resource, cpe, **kwargs):
        """
        :param resource: Vulnerable resource.
        :type resource: Resource

        :param cpe: CPE name of the outdated software.
        :type cpe: str

        """

        # Validate the parameters.
        if not isinstance(resource, Resource):
            raise TypeError("Expected Resource, got %r instead" % type(resource))
        if type(cpe) is not str:
            raise TypeError("Expected str, got %r instead" % type(cpe))
        if not cpe.startswith("cpe:"):
            raise ValueError("Not a CPE name: %r" % cpe)

        # Convert CPE <2.3 (URI binding) to CPE 2.3 (formatted string binding).
        if cpe.startswith("cpe:/"):
            cpe_parts = cpe[5:].split(":")
            if len(cpe_parts) < 11:
                cpe_parts.extend( "*" * (11 - len(cpe_parts)) )
            cpe = "cpe:2.3:" + ":".join(cpe_parts)

        # Vulnerable resource.
        self.__resource_id = resource.identity

        # CPE tag of the outdated software.
        self.__cpe = cpe

        # Parent constructor.
        super(OutdatedSoftware, self).__init__(**kwargs)

        # Link the vulnerability to the resource.
        self.add_resource(resource)

    __init__.__doc__ += Vulnerability.__init__.__doc__


    #--------------------------------------------------------------------------
    @identity
    def resource_id(self):
        """
        :returns: Identity hash of the vulnerable resource.
        :rtype: str
        """
        return self.__resource_id


    #--------------------------------------------------------------------------
    @property
    def resource(self):
        """
        :returns: Vulnerable resource.
        :rtype: Resource
        """
        return self.resolve(self.resource_id)


    #--------------------------------------------------------------------------
    @identity
    def cpe(self):
        """
        :returns: CPE 2.3 name of the outdated software.
        :rtype: str
        """
        return self.__cpe
