#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__ = """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2013

Authors:
  Daniel Garcia Garcia a.k.a cr0hn | cr0hn<@>cr0hn.com
  Mario Vilas | mvilas<@>gmail.com

Golismero project site: https://github.com/golismero
Golismero project mail: golismero.project<@>gmail.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

__all__ = ["DomainDisclosure"]

from .. import Vulnerability
from ... import identity

from ....net.web_utils import split_hostname
from ....text.text_utils import to_utf8

from netaddr import IPAddress


#------------------------------------------------------------------------------
class DomainDisclosure(Vulnerability):
    """
    Domain Disclosure.

    A domain was found by brute force or leaked by other means.
    """

    DEFAULTS = Vulnerability.DEFAULTS.copy()
    DEFAULTS["level"] = "informational"
    DEFAULTS["cwe"]   = "CWE-200"
    DEFAULTS["cvss_base"] = "2.2"
    DEFAULTS["references"] = (
        "https://www.owasp.org/index.php/Information_Leakage",
    )


    #--------------------------------------------------------------------------
    def __init__(self, hostname, **kwargs):
        """
        :param hostname: Domain name.
        :type hostname: str
        """

        # Sanitize the hostname.
        hostname = to_utf8(hostname)
        if not isinstance(hostname, str):
            raise TypeError(
                "Expected string, got %r instead" % type(hostname))

        # Check we've not confused an IP address for a hostname.
        try:
            if hostname.startswith("[") and hostname.endswith("]"):
                IPAddress(hostname[1:-1], version=6)
            else:
                IPAddress(hostname)
        except Exception:
            pass
        else:
            raise ValueError(
                "This is an IP address (%s) not a domain!" % hostname)

        # Set the properties.
        self.__hostname = hostname

        # Parent constructor.
        super(DomainDisclosure, self).__init__(**kwargs)

    __init__.__doc__ += Vulnerability.__init__.__doc__


    #--------------------------------------------------------------------------
    @identity
    def hostname(self):
        """
        :return: Domain name.
        :rtype: str
        """
        return self.__hostname


    #--------------------------------------------------------------------------
    @property
    def root(self):
        """
        :return: Root domain. i.e: www.mysite.com -> mysite.com
        :rtype: str
        """
        _, domain, suffix = split_hostname(self.hostname)
        if suffix:
            return "%s.%s" % (domain, suffix)
        return domain
