#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__= """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2013

Authors:
  Daniel Garcia Garcia a.k.a cr0hn | cr0hn<@>cr0hn.com
  Mario Vilas | mvilas<@>gmail.com

Golismero project site: https://github.com/golismero
Golismero project mail: golismero.project<@>gmail.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

__all__ = ["SuspiciousHTML"]

from .. import Vulnerability
from ... import identity
from ...information.html import HTML


#------------------------------------------------------------------------------
class SuspiciousHTML(Vulnerability):
    """
    Suspicious HTML Content.

    HTML content was found that may contain sensitive information.
    User attention could be required.
    """

    DEFAULTS = Vulnerability.DEFAULTS.copy()
    DEFAULTS["level"] = "informational"
    DEFAULTS["cvss_base"] = "2.9"
    DEFAULTS["references"] = (
        "https://www.owasp.org/index.php/Information_Leakage",
    )


    #--------------------------------------------------------------------------
    def __init__(self, content, substring, **kwargs):
        """
        :param content: HTML content where the suspicious substring was found.
        :type content: HTML

        :param substring: Substring that causes suspicion.
        :type substring: str

        """

        # Validate the data types.
        if not isinstance(content, HTML):
            raise TypeError("Expected HTML, got %r instead" % type(content))
        if type(substring) is not str:
            raise TypeError("Expected str, got %r instead" % type(substring))

        # Save the properties.
        self.__content_id = content.identity
        self.__substring  = substring

        # Parent constructor.
        super(SuspiciousHTML, self).__init__(**kwargs)

        # Associate the content to the vulnerability.
        self.add_information(content)

    __init__.__doc__ += Vulnerability.__init__.__doc__


    #--------------------------------------------------------------------------
    @identity
    def content_id(self):
        """
        :returns: Identity hash of the HTML content.
        :rtype: str
        """
        return self.__content_id


    #--------------------------------------------------------------------------
    @property
    def content(self):
        """
        :returns: HTML content where the suspicious substring was found.
        :rtype: HTML
        """
        return self.resolve(self.content_id)


    #--------------------------------------------------------------------------
    @identity
    def substring(self):
        """
        :returns: Substring that causes suspicion.
        :rtype: str
        """
        return self.__substring
