#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__= """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2013

Authors:
  Daniel Garcia Garcia a.k.a cr0hn | cr0hn<@>cr0hn.com
  Mario Vilas | mvilas<@>gmail.com

Golismero project site: https://github.com/golismero
Golismero project mail: golismero.project<@>gmail.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

__all__ = ["MaliciousUrl", "MaliciousDomain", "MaliciousIP", "MaliciousASN"]

from . import Malware
from .. import Vulnerability, UrlVulnerability, DomainVulnerability, \
    IPVulnerability
from ... import identity


#------------------------------------------------------------------------------
class MaliciousUrl(UrlVulnerability):
    """
    Malicious URL Detected.

    A URL was found that could contain output links to a malicious site or
    malware. This may be the result of a security intrusion, or a successful
    persistent XSS attack by a nefarious entity.

    You should review your website and ensure that your site was not
    compromised by a security intrusion.
    """

    DEFAULTS = Malware.DEFAULTS.copy()
    DEFAULTS["cvss_base"] = "4.3"

#------------------------------------------------------------------------------
class MaliciousDomain(DomainVulnerability):
    """
    Malicious Domain Detected.

    A domain was found that could contain output links to a malicious site or
    malware. This may be the result of a security intrusion, or a successful
    DNS poisoning attack by a nefarious entity.

    You should review your website and ensure that your site was not
    compromised by a security intrusion.
    """

    DEFAULTS = Malware.DEFAULTS.copy()
    DEFAULTS["cvss_base"] = "4.3"

#------------------------------------------------------------------------------
class MaliciousIP(IPVulnerability):
    """
    Malicious IP Detected.

    An IP was found that could contain output links to a malicious site or
    malware. This may be the result of a security intrusion, or a successful
    persistent XSS attack by a nefarious entity.

    You should review your website and ensure that your site was not
    compromised by a security intrusion.
    """

    DEFAULTS = Malware.DEFAULTS.copy()
    DEFAULTS["cvss_base"] = "5.8"

#------------------------------------------------------------------------------
class MaliciousASN(Vulnerability):
    """
    Malicious ASN Detected.

    An Autonomous System Number (ASN) was found that could contain a malicious
    site or malware. This may be the result of a security intrusion, or a
    successful BGP reconfiguration attack by a nefarious entity.

    You should review your website and ensure that your site was not
    compromised by a security intrusion.
    """
    DEFAULTS = Vulnerability.DEFAULTS.copy()
    DEFAULTS["cvss_base"] = "6.8"

    #--------------------------------------------------------------------------
    def __init__(self, asn, **kwargs):
        """
        :param asn: ASN where the vulnerability was found.
        :type asn: ASN

        """

        # Save the target object ID.
        self.__asn_id = asn.identity

        # Parent constructor.
        super(MaliciousASN, self).__init__(**kwargs)

        # Link the vulnerability to the target.
        self.add_information(asn)

    __init__.__doc__ += Vulnerability.__init__.__doc__


    #--------------------------------------------------------------------------
    @identity
    def asn_id(self):
        """
        :returns: Identity hash of the ASN
            where the vulnerability was found.
        :rtype: str
        """
        return self.__asn_id


    #--------------------------------------------------------------------------
    @property
    def asn(self):
        """
        :returns: ASN where the vulnerability was found.
        :rtype: ASN
        """
        return self.resolve(self.asn_id)
