#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__= """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2013

Authors:
  Daniel Garcia Garcia a.k.a cr0hn | cr0hn<@>cr0hn.com
  Mario Vilas | mvilas<@>gmail.com

Golismero project site: https://github.com/golismero
Golismero project mail: golismero.project<@>gmail.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

__all__ = ["DefaultErrorPage"]

from .. import Vulnerability, UrlVulnerability
from ... import identity

from posixpath import split


#------------------------------------------------------------------------------
class DefaultErrorPage(UrlVulnerability):
    """
    Default Error Page.

    A default error page was found in a web server.
    This may help an adversary to gather information prior to an attack.

    The details on how to override the default error pages with custom pages
    is specific to the HTTP server being used. Please consult the documentation
    of your HTTP server software on how to do this.
    """

    DEFAULTS = UrlVulnerability.DEFAULTS.copy()
    DEFAULTS["level"] = "informational"
    DEFAULTS["cwe"]   = "CWE-756"
    DEFAULTS["cvss_base"] = "1.6"
    DEFAULTS["references"] = (
        "https://www.owasp.org/index.php/Information_Leakage",
    )


    #--------------------------------------------------------------------------
    def __init__(self, url, server_name, **kwargs):
        """
        :param url: URL where the default error page was found.
        :type url: Url

        :param server_name: The web server that has this error page: Apache, IIS, Tomcat...
        :type server_name: str

        """

        # Name of the server that has this error page.
        self.__server_name = server_name

        # Parent constructor.
        super(DefaultErrorPage, self).__init__(url, **kwargs)

    __init__.__doc__ += Vulnerability.__init__.__doc__


    #--------------------------------------------------------------------------
    @identity
    def server_name(self):
        """
        :returns: Name of server that has this error page.
        :rtype: str
        """
        return self.__server_name


    #--------------------------------------------------------------------------
    @property
    def discovered_path(self):
        """
        :returns: Discovered part of the URL.
        :rtype: str
        """
        return split(self.url.parsed_url.path)[1]
