#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__= """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2013

Authors:
  Daniel Garcia Garcia a.k.a cr0hn | cr0hn<@>cr0hn.com
  Mario Vilas | mvilas<@>gmail.com

Golismero project site: https://github.com/golismero
Golismero project mail: golismero.project<@>gmail.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

__all__ = ["VulnerableService"]

from .. import Vulnerability
from ... import identity
from ...resource.ip import IP


#------------------------------------------------------------------------------
class VulnerableService(Vulnerability):
    """
    Vulnerable Service.

    A vulnerable service was found.

    Apply all missing patches or upgrade to a newer version.
    """

    DEFAULTS = Vulnerability.DEFAULTS.copy()
    DEFAULTS["level"] = "critical"
    DEFAULTS["cvss_base"] = "9"
    DEFAULTS["references"] = (
        "https://www.owasp.org/index.php/Top_10_2013-A5-Security_Misconfiguration",
    )


    #--------------------------------------------------------------------------
    def __init__(self, ip, port, protocol = "TCP", **kwargs):
        """
        :param ip: IP address.
        :type ip: IP

        :param port: Port number.
        :type port: int

        :param protocol: Protocol name ("TCP" or "UDP").
        :type protocol: str

        """

        # Validate the parameters.
        if not isinstance(ip, IP):
            raise TypeError("Expected IP, got %r instead" % type(ip))
        try:
            port = int(port)
        except Exception:
            raise TypeError("Expected int, got %r instead" % type(port))
        if not 0 < port < 65536:
            raise ValueError("Bad port number: %d" % port)
        if type(protocol) is not str:
            raise TypeError("Expected str, got %r instead" % type(protocol))
        protocol = protocol.upper()
        if protocol not in ("TCP", "UDP"):
            raise ValueError("Protocol not supported: %r" % protocol)

        # Save the properties.
        self.__target_id = ip.identity
        self.__address   = ip.address
        self.__port      = port
        self.__protocol  = protocol

        # Parent constructor.
        super(VulnerableService, self).__init__(**kwargs)

        # Link the vulnerability to the resource.
        self.add_resource(ip)

    __init__.__doc__ += Vulnerability.__init__.__doc__


    #--------------------------------------------------------------------------
    @identity
    def target_id(self):
        """
        :returns: Identity hash of the vulnerable resource.
        :rtype: str
        """
        return self.__target_id


    #--------------------------------------------------------------------------
    @property
    def target(self):
        """
        :returns: Vulnerable resource.
        :rtype: IP
        """
        return self.resolve(self.target_id)


    #--------------------------------------------------------------------------
    @identity
    def address(self):
        """
        :returns: IP address where the vulnerable service was found.
        :rtype: str
        """
        return self.__address


    #--------------------------------------------------------------------------
    @identity
    def port(self):
        """
        :returns: TCP or UDP port where the vulnerable service was found.
        :rtype: int
        """
        return self.__port


    #--------------------------------------------------------------------------
    @identity
    def protocol(self):
        """
        :returns: Protocol where the vulnerable service was found
            ("TCP" or "UDP").
        :rtype: str
        """
        return self.__protocol
