# -*- coding: utf-8 -*-

"""
tcp:
    传输的数据包格式:raw_data\r\n
udp:
    传输的数据包格式:raw_data
    因为并不通过末尾的符号做识别

解析之后的json:
{
    endpoint: 'get_name'        str
    **custom data
}

view_func 格式:
def func(request):
    pass

"""

import logging
import time
import json
import asyncore
import asynchat
import socket
import constants

logger = logging.getLogger('gourd')


class RouteMgr(object):
    """
    专门做路由管理
    """

    _rule_map = None

    def __init__(self):
        self._rule_map = dict()

    def add_url_rule(self, endpoint=None, view_func=None, **options):
        assert view_func is not None, 'expected view func if endpoint is not provided.'

        endpoint = endpoint or view_func.__name__
        if endpoint in self._rule_map and view_func != self._rule_map[endpoint]:
            raise Exception, 'repeat view_func for endpoint: %(endpoint)s, old_view_func:%(old_view_func)s, new_view_func: %(new_view_func)s' % dict(
                endpoint=endpoint,
                old_view_func=self._rule_map[endpoint].__name__,
                new_view_func=view_func.__name__,
            )

        self._rule_map[endpoint] = view_func

    def route(self, **options):
        def decorator(f):
            endpoint = options.pop('endpoint', None)
            self.add_url_rule(endpoint, f, **options)
            return f
        return decorator

    def get_view_func(self, endpoint):
        return self._rule_map.get(endpoint)


class TcpServer(asyncore.dispatcher, RouteMgr):
    _host = None
    _port = None
    _backlog = None
    _terminator = None

    def __init__(self):
        # 把map变成私有的
        asyncore.dispatcher.__init__(self, map=dict())
        RouteMgr.__init__(self)

    def handle_accept(self):
        sock, address = self.accept()
        logger.debug('accept. socket:%s, address:%s', sock, address)
        TcpRequestHandler(self, address, self._terminator, sock, self._map)

    def _init_socket(self):
        self.create_socket(socket.AF_INET, socket.SOCK_STREAM)
        self.set_reuse_addr()
        self.bind((self._host, self._port))
        self.listen(self._backlog)

    def run(self, host, port, backlog=None, terminator=None):
        self._host = host
        self._port = port
        self._backlog = backlog or constants.DEFAULT_BACKLOG
        self._terminator = terminator or constants.DEFAULT_TERMINATOR

        self._init_socket()

        asyncore.loop(map=self._map)


class TcpRequestHandler(asynchat.async_chat):
    _server = None
    _address = None
    _raw_data = None
    _json_data = None

    def __init__(self, server, address, terminator, sock, map=None):
        asynchat.async_chat.__init__(self, sock, map)
        self._server = server
        self._address = address
        # 用json做格式，结尾加上个 \r\n 就可以了。
        self.set_terminator(terminator)

    def collect_incoming_data(self, data):
        # 直接用默认提供的方法就行
        self._collect_incoming_data(data)

    def found_terminator(self):
        self._raw_data = self._get_data()
        try:
            self._json_data = json.loads(self._raw_data)
        except Exception, e:
            logger.error('json parse fail. e: %s, recv_data: %s', e, self._raw_data)
            return

        logger.debug('jdata: %s', self._json_data)

        if not isinstance(self._json_data, dict):
            logger.error('json_data is not dict. json_data: %s', self._json_data)
            return

        if not self._json_data.get('endpoint'):
            logger.error('endpoint is not in json_data. json_data: %s', self._json_data)
            return

        endpoint = self._json_data['endpoint']
        view_func = self.server.get_view_func(endpoint)
        if not view_func:
            logger.error('endpoint is not valid. endpoint: %s', endpoint)
            return

        try:
            view_func(self)
        except Exception, e:
            logger.error('view_func raise exception. endpoint: %s, view_func: %s, e: %s',
                         endpoint, view_func.__name__, e)

    def handle_close(self):
        # 对端关闭链接，或者自己关闭链接时，调用
        asynchat.async_chat.handle_close(self)
        logger.debug('socket closed')

    @property
    def server(self):
        """
        获取当前server
        """
        return self._server

    @property
    def address(self):
        return self._address

    @property
    def raw_data(self):
        return self._raw_data

    @property
    def json_data(self):
        """
        获取获取的json数据
        """
        return self._json_data


class UdpServer(asyncore.dispatcher, RouteMgr):
    _address = None
    _raw_data = None
    _json_data = None

    _host = None
    _port = None

    def __init__(self):
        # 把map变成私有的
        asyncore.dispatcher.__init__(self, map=dict())
        RouteMgr.__init__(self)

    def handle_read(self):
        self._raw_data, self._address = self.socket.recvfrom(constants.DEFAULT_PACKAGE_SIZE)
        try:
            self._json_data = json.loads(self._raw_data)
        except Exception, e:
            logger.error('json parse fail. e: %s, recv_data: %s', e, self._raw_data)
            return

        logger.debug('jdata: %s', self._json_data)

        if not isinstance(self._json_data, dict):
            logger.error('json_data is not dict. json_data: %s', self._json_data)
            return

        if not self._json_data.get('endpoint'):
            logger.error('endpoint is not in json_data. json_data: %s', self._json_data)
            return

        endpoint = self._json_data['endpoint']
        view_func = self.server.get_view_func(endpoint)
        if not view_func:
            logger.error('endpoint is not valid. endpoint: %s', endpoint)
            return

        try:
            view_func(self)
        except Exception, e:
            logger.error('view_func raise exception. endpoint: %s, view_func: %s, e: %s',
                         endpoint, view_func.__name__, e)

    def _init_socket(self):
        self.create_socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.bind((self._host, self._port))

    def run(self, host, port):
        self._host = host
        self._port = port

        self._init_socket()

        asyncore.loop(map=self._map)

    def push(self, data):
        """
        发送数据，为了tcp一致
        """
        return self.socket.sendto(data, self._address)

    @property
    def server(self):
        """
        获取当前server
        """
        return self

    @property
    def address(self):
        return self._address

    @property
    def raw_data(self):
        return self._raw_data

    @property
    def json_data(self):
        """
        获取获取的json数据
        """
        return self._json_data

