# -*- coding: utf-8 -*-

"""
tcp:
    传输的数据包格式:raw_data\n
udp:
    传输的数据包格式:raw_data
    因为并不通过末尾的符号做识别

解析之后的json:
{
    endpoint: 'get_name'        str
    **custom data
}

view_func 格式:
def func(request):
    pass

"""

import logging
import time
import json
import asyncore
import asynchat
import socket
import functools
import constants
from base import CallBackMgr


def handle_input_data(request):
    """
    因为udp和tcp的处理实在一样，所以还是抽离出来吧
    """
    request.server.logger.debug('raw_data: %s', request.raw_data)

    try:
        request.json_data = json.loads(request.raw_data)
    except Exception, e:
        error = 'parse raw_data fail. e: %s, raw_data: %s' % (e, request.raw_data)
        request.server.logger.error(error)
        request.push(request.server.make_rsp(ret=constants.RET_SYSTEM, error=error))
        request.close()
        return None

    if not isinstance(request.json_data, dict):
        error = 'json_data is not dict. json_data: %s' % request.json_data
        request.server.logger.error(error)
        request.push(request.server.make_rsp(ret=constants.RET_SYSTEM, error=error))
        request.close()
        return None

    if not request.json_data.get('endpoint'):
        error = 'endpoint is not in json_data. json_data: %s' % request.json_data
        request.server.logger.error(error)
        request.push(request.server.make_rsp(ret=constants.RET_SYSTEM, error=error))
        request.close()
        return None

    endpoint = request.json_data['endpoint']
    view_func = request.server.get_route_view_func(endpoint)
    if not view_func:
        error = 'endpoint is not valid. endpoint: %s' % endpoint
        request.server.logger.error(error)
        request.push(request.server.make_rsp(ret=constants.RET_SYSTEM, error=error))
        request.close()
        return None

    # before_request_func_list
    for func in request.server.before_request_func_list:
        try:
            func(request)
        except Exception, e:
            request.server.logger.error('before_request_func_list raise exception. func: %s, e: %s, traceback: %s',
                                        func.__name__, e, __import__('traceback').format_exc())
    view_func_exc = None
    view_func_result = None

    try:
        view_func_result = view_func(request)
    except Exception, e:
        error = 'view_func raise exception. endpoint: %s, view_func: %s, e: %s, traceback: %s' % (
            endpoint, view_func.__name__, e, __import__('traceback').format_exc())
        request.server.logger.error(error)
        view_func_exc = e
        request.push(request.server.make_rsp(ret=constants.RET_SYSTEM,
                                             error=error if request.server.debug else constants.ERROR_INTERNAL))

    # after_request_func_list
    for func in request.server.after_request_func_list:
        try:
            # 异常优先，以为一旦有异常，肯定没返回
            func(request, view_func_exc or view_func_result)
        except Exception, e:
            request.server.logger.error('after_request_func_list raise exception. func: %s, view_func_result: %s, e: %s, traceback: %s',
                                        func.__name__, view_func_result, e, __import__('traceback').format_exc())

    # 如果view_func异常，就要关闭socket
    if view_func_exc:
        request.close()

    return view_func_result


class GourdTcp(asyncore.dispatcher, CallBackMgr):
    # 为了和同步版保持一致
    request_queue_size = None
    terminator = None
    logger = None
    request_handler_class = None

    def __init__(self, log_name=None, terminator=None, backlog=None, request_handler_class=None):
        # 把map变成私有的
        asyncore.dispatcher.__init__(self, map=dict())
        CallBackMgr.__init__(self)
        self.logger = logging.getLogger(log_name or constants.LOG_NAME)
        self.terminator = terminator or constants.TERMINATOR
        self.request_queue_size = backlog or constants.BACKLOG
        self.request_handler_class = request_handler_class or GourdTcpRequestHandler

    def handle_accept(self):
        sock, address = self.accept()
        self.logger.debug('accept. socket:%s, address:%s', sock, address)
        self.request_handler_class(self, address, self.terminator, sock, self._map)

    def run(self, host, port):
        self.create_socket(socket.AF_INET, socket.SOCK_STREAM)
        self.set_reuse_addr()
        self.bind((host, port))
        self.listen(self.request_queue_size)

        asyncore.loop(map=self._map)

    def make_rsp(self, *args, **kwargs):
        """
        生成rsp
        """
        return json.dumps(dict(*args, **kwargs)) + self.terminator


class GourdTcpRequestHandler(asynchat.async_chat):
    server = None
    client_address = None
    raw_data = None
    json_data = None

    def __init__(self, server, client_address, terminator, sock, map=None):
        asynchat.async_chat.__init__(self, sock, map)
        self.server = server
        self.client_address = client_address
        self.set_terminator(terminator)

        # create_request_func_list
        for func in self.server.create_request_func_list:
            try:
                func(self)
            except Exception, e:
                self.server.logger.error('create_request_func_list raise exception. func: %s, e: %s, traceback: %s',
                                         func.__name__, e, __import__('traceback').format_exc())

    def collect_incoming_data(self, data):
        # 直接用默认提供的方法就行
        self._collect_incoming_data(data)

    def found_terminator(self):
        # 这里会把incoming data 清空
        self.raw_data = self._get_data()
        handle_input_data(self)

    def close(self):
        # handle_close 只能处理对方关闭链接的情况，handle_close中也是调用了close
        # 对端关闭链接，或者自己关闭链接时，调用
        asynchat.async_chat.close(self)
        self.server.logger.debug('socket closed')

        # teardown_request_func_list
        for func in self.server.teardown_request_func_list:
            try:
                func(self)
            except Exception, e:
                self.server.logger.error('teardown_request_func_list raise exception. func: %s, e: %s, traceback: %s',
                                         func.__name__, e, __import__('traceback').format_exc())


class GourdUdp(asyncore.dispatcher, CallBackMgr):
    logger = None
    request_handler_class = None

    def __init__(self, log_name=None, request_handler_class=None):
        # 把map变成私有的
        asyncore.dispatcher.__init__(self, map=dict())
        CallBackMgr.__init__(self)
        self.logger = logging.getLogger(log_name or constants.LOG_NAME)
        self.request_handler_class = request_handler_class or GourdUdpRequestHandler

    def handle_read(self):
        raw_data, address = self.socket.recvfrom(constants.PACKAGE_SIZE)
        request = self.request_handler_class(self, address, raw_data)
        request.process()

    def writable(self):
        """
        因为发送数据是直接sendto了，所以可以减少很多写事件循环
        """
        return False

    def run(self, host, port):
        self.create_socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.bind((host, port))

        asyncore.loop(map=self._map)

    def make_rsp(self, *args, **kwargs):
        """
        生成rsp
        """
        return json.dumps(dict(*args, **kwargs))


class GourdUdpRequestHandler(object):
    server = None
    client_address = None
    raw_data = None
    json_data = None

    def __init__(self, server, client_address, raw_data):
        super(GourdUdpRequestHandler, self).__init__()
        self.server = server
        self.client_address = client_address
        self.raw_data = raw_data

    def process(self):
        # create_request_func_list
        for func in self.server.create_request_func_list:
            try:
                func(self)
            except Exception, e:
                self.server.logger.error('create_request_func_list raise exception. func: %s, e: %s, traceback: %s',
                                         func.__name__, e, __import__('traceback').format_exc())
        handle_input_data(self)

        # teardown_request_func_list
        for func in self.server.teardown_request_func_list:
            try:
                func(self)
            except Exception, e:
                self.server.logger.error('teardown_request_func_list raise exception. func: %s, e: %s, traceback: %s',
                                         func.__name__, e, __import__('traceback').format_exc())

    def push(self, data):
        """
        发送数据，为了tcp一致
        """
        try:
            return self.server.socket.sendto(data, self.client_address)
        except socket.error, e:
            self.server.logger.error('socket.error. e:%s', e)
        except Exception, e:
            self.server.logger.error('exception occur. e: %s, tbk: %s', e, __import__('traceback').format_exc())

    def close(self):
        # 只是为了保持接口一致而已
        pass
