#!/usr/bin/env python
# encoding: utf-8
'''
bin.gus_investigation_history -- Creates investigation timelines for the last month or two

bin.gus_investigation_history is a utility that examines investigations created by support
    and shows the handoffs between teams in an HTML visualization

It defines classes_and_methods

@author:     scrosby

@copyright:  2014 salesforce.com. All rights reserved.

@license:    license

@contact:    scrosby@salesforce.com
@deffield    updated: 2014-09-27
'''

import sys, os, traceback, datetime

from optparse import OptionParser
from gus.SupportClient import SupportClient

__all__ = []
__version__ = 0.1
__date__ = '2014-09-26'
__updated__ = '2014-09-26'

class Segment:
    def __init__(self, start, end):
        self.start=start
        self.end=end
        
    def intersects(self, start):
        return self.start <= start and self.end >= start
    
    def hours(self):
        return (self.end - self.start).total_seconds() / 3600

def main(argv=None):
    '''Command line options.'''

    program_name = os.path.basename(sys.argv[0])
    program_version = "v0.1"
    program_build_date = "%s" % __updated__

    program_version_string = '%%prog %s (%s)' % (program_version, program_build_date)
    #program_usage = '''usage: spam two eggs''' # optional - will be autogenerated by optparse
    program_longdesc = '''''' # optional - give further explanation about what the program does
    program_license = "Copyright 2014 scrosby (salesforce.com)                                            \
                Licensed under the Apache License 2.0\nhttp://www.apache.org/licenses/LICENSE-2.0"

    if argv is None:
        argv = sys.argv[1:]
    try:
        # setup option parser
        parser = OptionParser(usage="%prog [options] support_user_id1 support_user_id2 ...", version=program_version_string, epilog=program_longdesc, description=program_license)
        parser.add_option("-f","--filter",dest="filter",action="append", help="Filter by touched teams, repeat for multiple teams")
        parser.add_option("-l","--last_month",dest="last_month",action="store_true", help="Include Previous month")

        # process options
        (opts, args) = parser.parse_args(argv)
        template = '''
<script type="text/javascript" src="https://www.google.com/jsapi?autoload={'modules':[{'name':'visualization',
       'version':'1','packages':['timeline']}]}"></script>
<script type="text/javascript">
google.setOnLoadCallback(makeChart);

function makeChart() {
    var container = document.getElementById('canvas');
    
    var chart = new google.visualization.Timeline(container);
    
    var dataTable = new google.visualization.DataTable();
    dataTable.addColumn({ type: 'string', id: 'Work' });
    dataTable.addColumn({ type: 'string', id: 'Assignment' });
    dataTable.addColumn({ type: 'date', id: 'Start' });
    dataTable.addColumn({ type: 'date', id: 'End' });
    
%s

    dataTable.sort(2);
    
    var options = {
            timeline: {
                groupByRowLabel: true,
                colorByRowLabel: true,
            },
            avoidOverlappingGridLines: true,
    };
    
    
    chart.draw(dataTable, options);
}

</script>
<div id="title">%s</div>
<div id="canvas" style="width: 3000px; height: 600px;"></div>
<div id="stats">%s</div>
        '''

        #SUPPORT_TEAM='00GB0000000t8wY'
        gus = SupportClient()
        out = []
        investigations = []
        names = []
        for arg in args:
            investigations.extend(gus.get_investigations_this_month_created_by_user(arg))
            if opts.last_month:
                investigations.extend(gus.get_investigations_last_month_created_by_user(arg))
            user = gus.get_chatter_profile(arg)
            names.append(user['name'])
        # remove duplicates
        investigations = set(investigations)
        
        time_cutoff = datetime.datetime.now().replace(day=1, hour=0, minute=0, second=0)
        if opts.last_month:
            time_cutoff = (time_cutoff - datetime.timedelta(days=1)).replace(day=1)
            
        total_handoffs=0
        total_filtered_handoffs=0
        total_filtered_investigations=0
        total_investigation_seconds=gus.get_seconds_for_work_id_list(investigations, period_start=time_cutoff)
        total_filtered_investigation_seconds=0
        segments = []
        TIME_FORMAT="%Y-%m-%dT%H:%M:%S.000+0000"
        for investigation in investigations:
            changes = gus.get_investigation_team_assignment_history(investigation)
            total_handoffs += (len(changes) - 1)
            filtered = change_filter(opts.filter, changes)
            if len(filtered) > 0:
                total_filtered_investigations += 1
                total_filtered_handoffs += (len(changes) - 1)
                work = gus.get_work_record(investigation)
                work_label="{} ({} handoffs)".format(work['Name'], len(changes) - 1)
                startdate = work['CreatedDate']
                for change in changes:
                    assignment = change[1]
                    if assignment is not None:
                        assignment = assignment.replace("'","\\'")
    
                        # print the row
                        enddate = change[0]
                        start = datetime.datetime.strptime(startdate, TIME_FORMAT)
                        if start < time_cutoff:
                            start = time_cutoff
                            startdate = datetime.datetime.strftime(start, TIME_FORMAT)
                        end = datetime.datetime.strptime(enddate, TIME_FORMAT)
                        if end < start:
                            end = start
                            enddate = startdate
                        
                        out.append("dataTable.addRows([['{}','{}',new Date(Date.parse('{}')),new Date(Date.parse('{}'))]])".format(work_label, assignment, startdate, enddate))
                        total_filtered_investigation_seconds += (end - start).total_seconds()
                        startdate = change[0]
                        assignment = change[2]
                        if assignment is not None:
                            assignment = assignment.replace("'","\\'")
                        
                if work['Closed__c'] == 1 and startdate is not None:
                    enddate = work['Closed_On__c']
                    # just in case the work is closed before the last assignment change
                    start = datetime.datetime.strptime(startdate, TIME_FORMAT)
                    if start < time_cutoff:
                        start = time_cutoff
                        startdate = datetime.datetime.strftime(start, TIME_FORMAT)
                    end = datetime.datetime.strptime(enddate, TIME_FORMAT)
                    if end < start:
                        end = start
                        enddate = startdate
                    if assignment is None:
                        assignment = work['Scrum_Team_Name__c']
                    assignment = "{} (Closed)".format(assignment)
                    out.append("dataTable.addRows([['{}','{}',new Date(Date.parse('{}')),new Date(Date.parse('{}'))]])".format(work_label, assignment, startdate, enddate))
                    total_filtered_investigation_seconds += (end - start).total_seconds()
                else:
                    start = datetime.datetime.strptime(startdate, TIME_FORMAT)
                    if start < time_cutoff:
                        start = time_cutoff
                        startdate = datetime.datetime.strftime(start, TIME_FORMAT)
                    enddate = datetime.datetime.strftime(datetime.datetime.utcnow(), TIME_FORMAT)
                    if assignment is None:
                        assignment = work['Scrum_Team_Name__c']
                    out.append("dataTable.addRows([['{}','{}',new Date(Date.parse('{}')),new Date(Date.parse('{}'))]])".format(work_label, assignment, startdate, enddate))
                    total_filtered_investigation_seconds += (end - start).total_seconds()

                investigation_start = datetime.datetime.strptime(work['CreatedDate'], TIME_FORMAT)
                if investigation_start < time_cutoff:
                    investigation_start = time_cutoff
                segments.append(Segment(investigation_start, end))
        statstable = '''
<table>
<tr><td>Total Investigations</td><td>{investigations}</td></tr>
<tr><td>Total Handoffs</td><td>{handoffs}</td></tr>
<tr><td>Avg Handoff Rate</td><td>{handoff_rate} / Investigation</td></tr>
<tr><td>Total Filtered Investigations</td><td>{filtered_investigations}</td></tr>
<tr><td>Total Filtered Handoffs</td><td>{filtered_handoffs}</td></tr>
<tr><td>Avg Handoff Rate (filtered)</td><td>{filtered_handoff_rate} / Investigation</td></tr>
<tr><td>Total Investigation Hours</td><td>{investigation_hours}</td></tr>
<tr><td>Average Hours per Investigation</td><td>{avg_hours_per_investigation}</td></tr>
<tr><td>Filtered Investigation Hours</td><td>{filtered_hours}</td></tr>
<tr><td>Average Hours per Filtered Investigation</td><td>{avg_filtered_hours_per_investigation}</td></tr>
<tr><td>Max Concurrent Investigations</td><td>{max_intersects}</td></tr>
<tr><td>Average Concurrent Investigations</td><td>{avg_intersects}</td></tr>
</table>
        '''
        intersect_counts = []
        segment_hours = []
        for s in segments:
            segment_hours.append(s.hours())
            count = 0
            for t in segments:
                if t.intersects(s.start):
                    count += 1
            intersect_counts.append(count)
            
        print("Total Investigations: {}".format(len(investigations)))
        print("Total Handoffs: {}".format(total_handoffs))
        print("Average Handoffs per Investigation: {}".format(float(total_handoffs) / float(len(investigations))))
        print("Total Hours: {}".format(total_investigation_seconds / 3600))
        if opts.filter:
            print("Total Filtered Investigations: {}".format(total_filtered_investigations))
            print("Total Filtered Handoffs: {}".format(total_filtered_handoffs))
            print("Average Handoffs per Filtered Investigation: {}".format(float(total_filtered_handoffs) / float(total_filtered_investigations)))
            print("Total Hours for {}: {}".format(", ".join(opts.filter), total_filtered_investigation_seconds / 3600))
            print("Average Hours for Filtered Investigations: {}".format((total_filtered_investigation_seconds / 3600) / total_filtered_investigations))
        print("Max Concurrent: {}".format(max(intersect_counts)))
        print("Average Concurrent: {}".format(sum(intersect_counts)/len(intersect_counts)))
        print("Average Hours per Investigation: {}".format((total_investigation_seconds / 3600) / len(investigations)))
        print("Maximum Investigation Time: {}".format(max(segment_hours)))
        report_name = "{}_Investigations.html".format(", ".join(names))
        with open(report_name, 'wb') as report:
            datatable = "\n".join(out)
            title = "Investigations Created By {}".format(", ".join(names))
            if opts.filter:
                title += "<br />Filtering Investigations Touched by {}".format(", ".join(opts.filter))
            stats = statstable.format(investigations=len(investigations), \
                                      handoffs=total_handoffs, \
                                      filtered_investigations=total_filtered_investigations, \
                                      filtered_handoffs=total_filtered_handoffs, \
                                      max_intersects=max(intersect_counts), \
                                      avg_intersects=sum(intersect_counts) / len(intersect_counts), \
                                      investigation_hours=total_investigation_seconds / 3600, \
                                      filtered_hours=total_filtered_investigation_seconds / 3600, \
                                      handoff_rate=round(float(total_handoffs) / float(len(investigations)), 2), \
                                      filtered_handoff_rate=round(float(total_filtered_handoffs) / float(total_filtered_investigations), 2), \
                                      avg_hours_per_investigation=round(float(total_investigation_seconds / 3600) / float(len(investigations)), 2), \
                                      avg_filtered_hours_per_investigation=round(float(total_filtered_investigation_seconds / 3600) / float(total_filtered_investigations), 2))
            render = template % (datatable, title, stats)
            report.write(render)
            report.close()
            
    except Exception as e:
        traceback.print_exc()
        indent = len(program_name) * " "
        sys.stderr.write(program_name + ": " + repr(e) + "\n")
        sys.stderr.write(indent + "  for help use --help")
        return 2

def change_filter(filter_list, change_list):
    if filter_list:
        out = []
        for item in change_list:
            if item[2] in filter_list:
                out.append(item)
                
        return out
    else:
        return change_list

if __name__ == "__main__":
    sys.exit(main())