#!/usr/bin/env python
# encoding: utf-8

import sys
import os
from argparse import ArgumentParser
from gus.BacklogClient import BacklogClient

__all__ = []
__version__ = 0.1
__date__ = '2013-11-28'
__updated__ = '2013-11-28'

gus = BacklogClient()

def assign(work, sprint):
    print("Assigning {} to sprint {}".format(work, sprint))
    gus.assign_work_to_sprint(work, sprint)

def onboard_p1(teamid, sprintid):
    p1s = gus.get_open_p1_bugs_for_team(teamid)
    print("Onboarding {} Open P1 Bugs".format(len(p1s)))
    for work in p1s:
        assign(work, sprintid)

def onboard_p2(teamid, sprintid):
    p2s = gus.get_open_p2_bugs_for_team(teamid)
    print("Onboarding {} Open P2 Bugs".format(len(p2s)))
    for work in p2s:
        assign(work, sprintid)

def pull_work_forward(last_sprint, this_sprint):
    incomplete = gus.get_open_work_for_sprint(last_sprint)
    print("Assigning {} incomplete items from last sprint to the current sprint".format(len(incomplete)))
    for work in incomplete:
        assign(work, this_sprint)

def push_work_forward(this_sprint, next_sprint):
    unresolved = gus.get_unresolved_work_for_sprint(this_sprint)
    print("Assigning {} unresolved items from this sprint to next sprint".format(len(unresolved)))
    for work in unresolved:
        assign(work, next_sprint)

def next_sprint(teamid):
    sprint = gus.get_next_future_sprint_for_team(teamid)
    push_work_forward(sprint)

def this_sprint(teamid):
    last_sprint = gus.get_previous_sprint_for_team(teamid)
    this_sprint = gus.get_current_sprint_for_team(teamid)
    sprint_record = gus.get_sprint_record(this_sprint)
    print("Setting up Sprint {}".format(sprint_record['Name']))
    pull_work_forward(last_sprint, this_sprint)
    onboard_p1(teamid, this_sprint)
    onboard_p2(teamid, this_sprint)

def main(argv=None): # IGNORE:C0111
    '''Command line options.'''
    
    if argv is None:
        argv = sys.argv
    else:
        sys.argv.extend(argv)

    program_name = os.path.basename(sys.argv[0])
    program_version = "v0.1"
    program_build_date = "%s" % __updated__
 
    program_version_string = '%s %s (%s)' % (program_name, program_version, program_build_date)
    program_longdesc = '''Updates Sprints with incomplete work'''
    program_license = "Copyright 2013 Shawn Crosby (salesforce.com)                                            \
                Licensed under the Apache License 2.0\nhttp://www.apache.org/licenses/LICENSE-2.0"
 
    if argv is None:
        argv = sys.argv[1:]
    try:
        # setup option parser
        parser = ArgumentParser(version=program_version_string, epilog=program_longdesc, description=program_license)
        parser.add_argument("-t", "--teamid", dest="teamid", help="Use the Object ID representing the team in GUS")
        args, unknown = parser.parse_known_args(argv)
        
        if args.teamid:
            this_sprint(args.teamid)
            return 0
        else:
            print "Need to specify a team id from GUS"
            return 2
    except KeyboardInterrupt:
        ### handle keyboard interrupt ###
        return 0
    except Exception as e:
        indent = len(program_name) * " "
        sys.stderr.write(program_name + ": " + repr(e) + "\n")
        sys.stderr.write(indent + "  for help use --help")
        return 2

if __name__ == "__main__":
    sys.exit(main())