from sf.Chatter import Chatter

class Cache:
    def __init__(self):
        self.cache = {}
        self.hitcount = 0
        self.misscount = 0
        
    def get(self, object_type, key):
        result = self.cache.get("{}_{}".format(object_type, key))
        if result is None:
            self.misscount += 1
        else:
            self.hitcount += 1
        return result
    
    def put(self, object_type, key, obj):
        self.set("{}_{}".format(object_type, key), obj)
    
    def set(self, key, obj):
        self.cache[key] = obj
            
class Client(Chatter):
    '''
    Base class facilitates logging into GUS and setting up a session and
    not much else.  Use BacklogClient to work with tickets or implement
    your own based on simple_salesforce methods.
    '''
    object_cache = Cache()
    WORK_RECORD_TYPES = {
                    "User Story":"0129000000006gDAAQ",
                    "Bug":"012T00000004MUHIA2",
                    "Investigation":"0129000000006lWAAQ",
                    "Todo":"0129000000006ByAAI",
                    }
    
    WORK_FREQUENCY = {
                      "Always": "a0L9000000000usEAA",
                      "Sometimes": "a0L9000000000urEAA",
                      "Often": "a0L9000000000utEAA",
                      "Rarely": "a0L9000000000uuEAA",
                      }
    
    WORK_IMPACT = {
                   "Malfunctioning": "a0O900000004EF2EAM",
                   }
    
    WORK_PRIORITY = {
                     "P0": "a0F900000008jfYEAQ",
                     "P1": "a0F900000008jfZEAQ",
                     "P2": "a0F900000008jfaEAA",
                     "P3": "a0F900000008jfbEAA",
                     "P4": "a0F900000008jfdEAA",
                     }
    
    def __init__(self, session_id=None):
        Chatter.__init__(self, instance='gus.salesforce.com', session_id=session_id)
        self.sf_session = self.client
    
    def session_id(self):
        '''
        Returns the Gus Session ID
        '''
        return self.session_id

    def get_user_id_for_email(self, email):
        '''
        Determines the object id of a user by looking up the record with the user's email address
        '''
        try:
            result = self.sf_session.query("select Id from User where Email = '%s'" % email)
            userid = result['records'][0]['Id']
        except:
            userid = None
        
        return userid
    
    def get_current_user_id(self):
        '''
        Determines the ID of the currently logged in user
        '''
        username = self.get_current_user()
        result = self.sf_session.query("select Id from User where Username='%s'" % username)
        return result['records'][0]['Id']
    
    def get_user_email(self, userid):
        '''
        Determines the email address of the specified user id
        '''
        result = self.sf_session.query("select Email from User where Id='%s'" % userid)
        return result['records'][0]['Email']
        
    def get_scrum_teams_for_user(self, userid):
        '''
        Returns a list of team id, team name tuples that the specified user is a member of with > 0% allocation
        '''
        email = self.get_user_email(userid)
        result = self.sf_session.query("select Scrum_Team__c, Scrum_Team_Name__c from ADM_Scrum_Team_Member__c where Internal_Email__c = '%s' and Allocation__c > 0" % email)
        out = []
        for record in result['records']:
            out.append((record['Scrum_Team__c'], record['Scrum_Team_Name__c']))
            
        return out
    
    def find_work(self, work_name):
        '''
        Returns the work record for the work specified by name
        '''
        result = self.sf_session.query("select Id from ADM_Work__c where Name='%s'" % work_name)
        try:
            work_id = result["records"][0]["Id"]
            work = self.get_work_record(work_id)
        except:
            raise NoRecordException('Can\'t find work ' + work_name)

        return work

    def get_work_record(self, work_id):
        '''
        Returns the record for the specified work record id
        '''
        if work_id is not None:
            result = self.object_cache.get("work", work_id)
            if result is None:
                result = self.sf_session.ADM_Work__c.get(work_id)
                self.object_cache.put("work", work_id, result)
        else:
            result = None
            
        return result
    
    def get_parent_work_for_work(self, workid):
        '''
        Returns a list of work record ids that are 'parents' to the specified work
        '''
        result = self.sf_session.query("Select Parent_Work__c from ADM_Parent_Work__c where Child_Work__c='%s'" % workid)
        return [x['Parent_Work__c'] for x in result['records']]
    
    def get_child_work_for_work(self, workid):
        '''
        Returns a list of work record ids that are 'children' to the specified work
        '''
        result = self.sf_session.query("Select Child_Work_c from ADM_Parent_Work__c where Parent_Work__c='%s'" % workid)
        return [x['Child_Work__c'] for x in result['records']]
    
    def get_sprint_for_work(self, workid):
        work = self.get_work_record(workid)
        if work['Sprint__c'] is not None:
            sprint = self.get_sprint_record(work['Sprint__c'])
        else:
            sprint = None
            
        return sprint
    
    def get_work_tree(self, workid):
        work = self.get_work_record(workid)
        work.parents = []
        parents = self.get_parent_work_for_work(workid)
        for parent in parents:
            work.parents.append(self.get_work_tree(parent))
            
        work.children = []
        children = self.get_child_work_for_work(workid)
        for child in children:
            work.children.append(self.get_work_tree(child))
            
        return work

    def find_build_id(self, build_name):
        '''
        Returns the id for a build specified by name
        '''
        result = self.sf_session.query("select Id from ADM_Build__c where Name='%s'" % build_name)
        try:
            build_id = result["records"][0]["Id"]
        except:
            raise NoRecordException('Can\'t find build ' + build_name)

        return build_id

    def get_current_sprint_for_team(self, teamid):
        '''
        Returns the Id of the current sprint for the specified team
        '''
        try:
            sprints = self.sf_session.query("select Id from ADM_Sprint__c where Scrum_Team__c = '%s' and Days_Remaining__c not in('CLOSED','NOT STARTED')" % teamid)
            sprint_id = sprints['records'][0]['Id']
        except:
            sprint_id = None
            
        return sprint_id
    
    def get_previous_sprint_for_team(self, teamid):
        try:
            sprints = self.sf_session.query("SELECT Id FROM ADM_Sprint__c WHERE Scrum_Team__c = '%s' AND Days_Remaining__c = 'CLOSED' ORDER BY Start_Date__c DESC NULLS FIRST" % teamid)
            sprint_id = sprints['records'][0]['Id']
        except:
            sprint_id = None
            
        return sprint_id
    
    def get_next_future_sprint_for_team(self, teamid):
        try:
            sprints = self.sf_session.query("SELECT Id FROM ADM_Sprint__c WHERE Scrum_Team__c = '%s' AND Days_Remaining__c = 'NOT STARTED' ORDER BY Start_Date__c ASC NULLS FIRST" % teamid)
            sprint_id = sprints['records'][0]['Id']
        except:
            sprint_id = None
            
        return sprint_id
    
    def get_team_record(self, teamid):
        '''
        Returns the record for a specified team id
        '''
        return self.__retrieve__("ADM_Scrum_Team__c", "team", teamid)
    
    def get_dependency_record(self, dependency_id):
        '''
        Returns the record for the specified dependency record id
        '''
        return self.__retrieve__("ADM_Team_Dependency__c", "dep", dependency_id)
    
    def get_build_record(self, build_id):
        '''
        Returns the build for the specified build record id
        '''
        return self.__retrieve__("ADM_Build__c", "build", build_id)
    
    def get_test_case_record(self, test_case_id):
        '''
        Returns a test case object
        '''
        return self.__retrieve__('QA_Test_Case__c', "test_case", test_case_id)
    
    def get_test_execution_record(self, test_exec_id):
        return self.__retrieve__('QA_Test_Execution__c', "test_exec", test_exec_id)
    
    def get_acceptance_criterion_record(self, ac_id):
        return self.__retrieve__('ADM_Acceptance_Criterion__c', "acceptance_criterion", ac_id)
    
    def get_sprint_record(self, sprint_id):
        '''
        Returns the record for the specified sprint record id
        '''
        return self.__retrieve__("ADM_Sprint__c", "sprint", sprint_id)
    
    def get_theme_record(self, themeid):
        return self.__retrieve__("ADM_Theme__c", "theme", themeid)
    
    def __retrieve__(self, object_name, tag_name, object_id):
        result = self.object_cache.get(tag_name, object_id)
        if result is None:
            m = getattr(self.sf_session, object_name)
            result = m.get(object_id)
            self.object_cache.put(tag_name, object_id, result)
        return result
    

class NoRecordException(Exception):
    """Thrown when record isn't found"""
    pass
