from gus.Gus import Client

class QAForceClient(Client):
    def find_test_case_by_ftest_id(self, ftest_id):
        '''
        Searches for gus test case using the FTest_Id__c field.  If it finds more than one test with the
        same ftest id, it raises an exception.
        
        @param ftest_id: presumably unique identifier for a test case
        @raise Exception: when a duplicate test case is found with the same ftest_id
        @return: ID of a test case matching the supplied FTest_id
        '''
        result = self.sf_session.query("Select Id from QA_Test_Case__c where Ftest_Id__c = '{}'".format(ftest_id))
        out = [x['Id'] for x in result['records']]
        if len(out) == 0:
            return None
        elif len(out) == 1:
            return out[0]
        else:
            raise Exception("Duplicate FTest ID used in Test Case: {}".format(ftest_id))
        
    def get_or_create_test_case_for_ftest(self, description, hierarchy, test_type, steps, ftest_id):
        '''
        Method searches for a test case by a given ftest_id.  If one can't be found, it creates a new
        one and returns it.
        
        @param description: title of test scenario to be used if not found
        @param hierarchy: hierarchy to place test case in library if not found
        @param test_type: type of test case to create if not found
        @param steps: Steps to be used for test case if not found
        @param ftest_id: ID used to find test case, used as key if not found
        @return: Full test case record either created or found
        '''
        test_case = self.find_test_case_by_ftest_id(ftest_id)
        if test_case is None:
            test_case = self.create_test_case(description, hierarchy, test_type, steps, ftest_id)
        return self.get_test_case_record(test_case)
    
    def get_or_create_test_execution_for_test_case(self, work_id, test_case):
        '''
        Method attempts to find a test execution for a given test case and user story.  If one
        can't be found it creates it automatically
        
        @param work_id: ID of the work item
        @param test_case: ID of the test case
        @return: Full record of test execution found or created
        '''
        test_exec = self.get_test_execution_for_test_case_and_story(work_id, test_case)
        if test_exec is None:
            test_exec = self.create_test_execution(test_case, work_id, 'Manual')
        return self.get_test_execution_record(test_exec)
    
    def get_test_case_ids_for_story(self, work_id):
        '''
        Returns a list of test case ids that are associated to a given user story
        
        @param work_id: ID of a user story to find test cases associated via test executions
        @return: array of test case ids
        '''
        result = self.sf_session.query("Select Test_Case__c from QA_Test_Execution__c where User_Story__c = '{}'".format(work_id))
        return [x['Test_Case__c'] for x in result['records']]
    
    def get_test_execution_for_test_case_and_story(self, work_id, test_case_id):
        '''
        Finds a unique execution record for a given story and test case.
        
        @param work_id: id of the user story tested
        @param test_case_id: id of the test case being executed
        @raise Exception: When two execution records for the same user story and test case combination are found
        @return: Id of the test execution
        '''
        result = self.sf_session.query("Select Id from QA_Test_Execution__c where User_Story__c = '{}' and Test_Case__c = '{}'".format(work_id, test_case_id))
        out = [x['Id'] for x in result['records']]
        if len(out) == 0:
            return None
        elif len(out) == 1:
            return out[0]
        else:
            raise Exception("Duplicate Test Execution: {}".format(test_case_id))
    
    def record_test_execution(self, work_id, test_case_id, build, e_date='TODAY', status='Passed', comment=None):
        '''
        Records the status of a test given the test case and user story, Creates new test execution record
        if required.
        
        @param work_id: Id of the work item that the test is executing for
        @param test_case_id: Id of the test case executed
        @param build: Name of the build being tested
        @param e_date: Date of execution
        @param status: Status of test (ie passed/failed)
        @param comment: comment to attache to test execution
        '''
        test_exec = self.get_or_create_test_execution_for_test_case(work_id, test_case_id)
        self.sf_session.QA_Test_Execution__c.update(test_exec['Id'], {'Build_of_Last_Execution__c': build, 'Type__c': 'Client Automation', 'Number_of_Executions__c': test_exec['Number_of_Executions__c'] + 1, 'Date_of_Last_Execution__c': e_date, 'Status__c': status, 'Comments__c': comment})
        
    def change_test_type(self, test_case_id, test_type):
        '''
        Change the execution type of a specified test
        
        @param test_case_id: ID of the test to change the execution type
        @param test_type: Test type value (ie Manual, Ftest, etc)
        '''
        self.sf_session.QA_Test_Case__c.update(test_case_id, {'Execution_Type__c': test_type})
        
    def create_test_case(self, description, hierarchy, test_type, steps, ftest_id=None):
        '''
        Creates a test case record
        
        @param description: Brief description of the test case
        @param hierarchy: Hierarchy to place test in the test case repository
        @param test_type: type of test (ie manual, ftest, selenium, etc)
        @param steps: Test steps, should be in Gherkin format
        @param ftest_id: String to identify the test
        @return: ID of the test case created
        '''
        test_case = self.sf_session.QA_Test_Case__c.create({'Brief_Description__c': description, 'Hierarchy__c': hierarchy, 'Execution_Type__c': test_type, 'Steps_to_Reproduce__c': steps, 'Ftest_Id__c': ftest_id})
        return test_case['id']

    def create_test_execution(self, test_case_id, work_id, execution_type='Client Automation'):
        '''
        Creates a test execution record for a test case and user story.  Default type is 'Client Automation'
        Assigns the test case to the current user and sets the status to 'Planned'
        
        @param test_case_id: ID of the test case to associate to
        @param work_id: ID of the work item to associate to
        @param execution_type: type of test execution
        '''
        user_id = self.get_current_user_id()
        test_execution = self.sf_session.QA_Test_Execution__c.create({'User_Story__c': work_id, 'Test_Case__c': test_case_id, 'Assignee__c': user_id, 'Type__c': execution_type, 'Status__c':'Planned'})
        return test_execution['id']
        
    def add_story_to_plan(self, plan_id, work_id):
        '''
        Adds a user story to a specified test plan
        
        @param plan_id: ID of the test plan to add the work to
        @param work_id: ID of the work item to add to the test plan
        '''
        self.sf_session.ADM_Work__c.update(work_id, {'Test_Plan__c': plan_id})
        
    def create_test_failure_bug(self, work_id, test_case_id, build_name, failure=None):
        '''
        Creates a Bug in GUS of type 'Test Failure' associated to a specified test case and user story
        using the user story as a template
        
        @param work_id: ID of the user story that the bug is found on
        @param test_case_id: ID of the test case that failed
        @param build_name: Name of the build that the test failed against
        @param failure: Failure message
        @return: ID of the bug created or existing if matching bug found
        '''
        user_story = self.get_work_record(work_id)
        test_execution = self.get_test_execution_for_test_case_and_story(work_id, test_case_id)
        test_execution_record = self.get_test_execution_record(test_execution)
        test_case = self.get_test_case_record(test_case_id)
        build_id = self.find_build_id(build_name)
        
        if failure is not None:
            fail_message = "[TEST FAILED] {}".format(failure)
        else:
            fail_message = "[TEST FAILED] {}".format(test_case["Brief_Description__c"])
        
        # if we have an open issue for this, just return it
        if test_execution_record['Bugs__c'] is not None:
            for bug in test_execution_record['Bugs__c'].split(","):
                test_bug = self.find_work(bug)
                if test_bug['Type__c'] == "Test Failure" and test_bug['Closed__c'] == 0 and test_bug['Subject__c'] == fail_message:
                    return test_bug['Id']
                
        description = "{}\n\n{}".format(test_case['Steps_to_Reproduce__c'], failure)
                
        #otherwise create a new one
        new_work = {
                    "Subject__c": fail_message,
                    "Status__c": "New",
                    "Assignee__c": user_story['Assignee__c'],
                    "Product_Owner__c": user_story['Product_Owner__c'],
                    "QA_Engineer__c": user_story['QA_Engineer__c'],
                    "RecordTypeId": self.WORK_RECORD_TYPES['Bug'],
                    "Type__c": "Test Failure",
                    "Found_in_Build__c": build_id,
                    "Impact__c": self.WORK_IMPACT['Malfunctioning'],
                    "Frequency__c": self.WORK_FREQUENCY['Always'],
                    "Priority__c": "P1",
                    "Details_and_Steps_to_Reproduce__c": description,
                    "Product_Tag__c": user_story['Product_Tag__c'],
                    }
        result = self.sf_session.ADM_Work__c.create(new_work)
        new_work_record = self.get_work_record(result['id'])
        
        # parent the user story to this bug
        parent_data = {
                       "Child_Work__c": result['id'],
                       "Parent_Work__c": work_id,
                       }
        self.sf_session.ADM_Parent_Work__c.create(parent_data)
        
        # update the test execution record
        bug_list = new_work_record['Name']
        if test_execution_record['Bugs__c'] is not None:
            bug_list = "{}, {}".format(test_execution_record['Bugs__c'], bug_list)
            
        self.sf_session.QA_Test_Execution__c.update(test_execution_record['Id'], {"Bugs__c": bug_list})
        
        return result['id']