from .Gus import Client
import re
from agile.StoryParser import StoryParser

class BacklogClient(Client):
    '''
    Module to interact with work on the backlog
    '''
    def mark_work_fixed(self, work_id, build_id):
        '''
        Sets status to fixed on the specified work item.  Requires a scheduled build when marking fixed
        '''
        self.sf_session.ADM_Work__c.update(work_id, {'Status__c': 'Fixed', 'Scheduled_Build__c': build_id})

    def mark_work_in_progress(self, work_id):
        '''
        Sets status of specified work item to 'In Progress'
        '''
        self.sf_session.ADM_Work__c.update(work_id, {'Status__c': 'In Progress'})

    def add_comment(self, work_id, comment):
        '''
        Adds a comment to the specified work item
        '''
        self.sf_session.ADM_Comment__c.create({
            'Work__c': work_id,
            'Body__c': comment,
        })
        
    def assign_work_to_sprint(self, work_id, sprint_id):
        self.sf_session.ADM_Work__c.update(work_id, {'Sprint__c': sprint_id})
        
    def assign_work_to_build(self, work_id, build_id):
        self.sf_session.ADM_Work__c.update(work_id, {'Scheduled_Build__c': build_id})
        
    def add_collab_link(self, work, link):
        '''
        Adds a link to a code review to the comment and related URL field on a work item if the related URL field is blank
        '''
        if work['Related_URL__c'] == '' or work['Related_URL__C'] is None:
            self.sf_session.ADM_Work__c.update(work['Id'], {'Related_URL__c': link})
            
        self.add_comment(work['Id'], 'Code Review Created: %s' % link)
        
    def get_open_work_for_user_id(self, user_id):
        '''
        Returns a list of work that is assigned to a specified user and not 'resolved'
        '''
        try:
            result = self.sf_session.query("select Name, Status__c, Subject__c from ADM_Work__c where Assignee__c = '%s' and Resolved__c = 0" % user_id)
            out = []
            for record in result['records']:
                out.append((record['Name'], record['Status__c'], record['Subject__c']))
        except:
            out = None
            
        return out
    
    def get_in_progress_work_for_user_id(self, user_id):
        '''
        Returns a list of work that is assigned to a specified user and status 'In Progress'
        '''
        try:
            result = self.sf_session.query("select Name, Status__c, Subject__c from ADM_Work__c where Assignee__c = '%s' and Status__c = 'In Progress'" % user_id)
            out = []
            for record in result['records']:
                out.append((record['Name'], record['Status__c'], record['Subject__c']))
        except:
            out = None
            
        return out
    
    def get_work_with_active_tasks_for_user(self, user_id):
        '''
        Returns a list of unresolved work items that have tasks assigned to the specified user where
        the tasks are not complete
        '''
        result = self.sf_session.query("select Work__c from ADM_Task__c where Assigned_To__c='%s' and Status__c!='Completed'"  % user_id)
        out = []
        for record in result['records']:
            if record['Work__c'] not in [x[0] for x in out]:
                data = self.sf_session.ADM_Work__c.get(record['Work__c'])
                if data['Resolved__c'] == 0:
                    out.append((data['Name'], data['Status__c'], data['Subject__c']))
            
        return out
    
    def get_open_work_for_sprint(self, sprintid):
        '''
        Returns a list of ids representing work that is assigned to a sprint and not closed.  The purpose of this
        method would be to gather any unfinished sprint work for transfer from the last sprint instead of having
        to modify each item individually
        '''
        result = self.sf_session.query("select Id from ADM_Work__c where Sprint__c='%s' and Closed__c = 0" % sprintid)
        return [x['Id'] for x in result['records']]
    
    def get_unresolved_work_for_sprint(self, sprintid):
        '''
        Returns a list of ids representing work that is assigned to a sprint but not yet resolved.  The idea is
        that the caller wants to get rid of stuff on the current sprint that won't get finished and likely set up
        the next sprint in advance
        '''
        result = self.sf_session.query("select Id from ADM_Work__c where Sprint__c='%s' and Resolved__c = 0" % sprintid)
        return [x['Id'] for x in result['records']]
    
    def get_unresolved_p1_bugs_for_team(self, teamid):
        '''
        Returns a list of ids that represents all the P1 bugs for a team that are not fixed and potentially targeted
        to an actual build so that they can be retargeted to another release
        '''
        result = self.sf_session.query("SELECT Id FROM ADM_Work__c WHERE Scrum_Team__c = '%s' AND Priority__c = 'P1' AND Resolved__c = 0 AND RecordTypeId = '012T00000004MUHIA2' and Status__c != 'Ready for Review'" % teamid)
        return [x['Id'] for x in result['records']]

    def get_open_p1_bugs_for_team(self, teamid):
        '''
        Returns a list of P1 bugs that are not yet closed so that they can perhaps be assigned en masse to the next
        sprint
        '''
        result = self.sf_session.query("SELECT Id FROM ADM_Work__c WHERE Scrum_Team__c = '%s' AND Priority__c = 'P1' AND Closed__c = 0 AND RecordTypeId = '012T00000004MUHIA2'" % teamid)
        return [x['Id'] for x in result['records']]
    
    def get_unresolved_p2_bugs_for_team(self, teamid):
        '''
        Returns a list of P2 bugs that are not in a fixed state and targeted to a specific build.  Likely so that
        the caller can target to the next release
        '''
        result = self.sf_session.query("SELECT Id FROM ADM_Work__c WHERE Scrum_Team__c = '%s' AND Priority__c = 'P2' AND Resolved__c = 0 AND RecordTypeId = '012T00000004MUHIA2' and Status__c != 'Ready for Review'" % teamid)
        return [x['Id'] for x in result['records']]

    def get_open_p2_bugs_for_team(self, teamid):
        '''
        Returns a list of P2 bugs that have not been closed, likely so that the caller can assign them en masse to
        the next sprint.
        '''
        result = self.sf_session.query("SELECT Id FROM ADM_Work__c WHERE Scrum_Team__c = '%s' AND Priority__c = 'P2' AND Closed__c = 0 AND RecordTypeId = '012T00000004MUHIA2'" % teamid)
        return [x['Id'] for x in result['records']]
        
    def get_work_for_sprint(self, sprintid):
        '''
        Returns a list of unresolved work in the specifed sprint
        '''
        result = self.sf_session.query("select Name, Status__c, Subject__c from ADM_Work__c where Sprint__c='%s' and Resolved__c = 0" % sprintid)
        out = []
        for record in result['records']:
            out.append((record['Name'], record['Status__c'], record['Subject__c']))
            
        return out
    
    def get_work_ids_for_sprint(self, sprintid):
        result = self.sf_session.query("Select Id from ADM_Work__c where Sprint__c='{}'".format(sprintid))
        return [x['Id'] for x in result['records']]
    
    def get_open_work_for_user(self, email):
        '''
        returns a list of open work by email instead of user id
        '''
        user_id = self.get_user_id_for_email(email)
        return self.get_open_work_for_user_id(user_id)
        
    def get_sprint_work_for_teams(self, user_id):
        '''
        Returns a list of unresolved work in the current sprint for all teams that a user is assigned to
        '''
        teams = self.get_scrum_teams_for_user(user_id)
        sprint_work = []
        for team in teams:
            current_sprint = self.get_current_sprint_for_team(team[0])
            if current_sprint is not None:
                team_sprint_work = self.get_work_for_sprint(current_sprint)
                for w in team_sprint_work:
                    sprint_work.append(w)
                    
        return sprint_work

    
    def get_potential_work_for_user(self, user_id):
        '''
        Returns a list of work that is: a) assigned to the specified user b) assigned to the current sprint for all
        teams that the use is a member of and c) has uncompleted tasks assigned to the user
        '''
        out = []
        # get in progress work
        in_progress = self.get_in_progress_work_for_user_id(user_id)
        
        # get work in sprint
        sprint_work = self.get_sprint_work_for_teams(user_id)

        # get work with tasks assigned
        task_work = self.get_work_with_active_tasks_for_user(user_id)
        
        for w in in_progress:
            out.append(w)
            
        for w in sprint_work:
            if w[0] not in [x[0] for x in out] and w[1] == 'In Progress':
                out.append(w)
        
        for w in task_work:
            if w[0] not in [x[0] for x in out]:
                out.append(w)
                
        return out
    
    def find_work_for_theme(self, themeid):
        '''
        Finds ids of work for a specified theme
        
        @param themeid: GUS ID for the theme
        @return: array of gus work ids
        '''
        result = self.sf_session.query("Select Work__c from ADM_Theme_Assignment__c where Theme__c = '{}'".format(themeid))
        return [x['Work__c'] for x in result['records']]
    
    def get_work_for_scheduled_build(self, build_id, team=None):
        '''
        Lists work ids for a specified build and team if supplied
        
        @param build_id: GUS ID of the build
        @param team: GUS ID of team
        @return: array of work ids in specified build
        '''
        if team is not None:
            result = self.sf_session.query("Select Id from ADM_Work__c where Scheduled_Build__c = '{}' and Scrum_Team__c = '{}'".format(build_id, team))
        else:
            result = self.sf_session.query("Select Id from ADM_Work__c where Scheduled_Build__c = '{}'".format(build_id))
        return [x['Id'] for x in result['records']]
    
    def get_work_ids_for_release(self, release, team=None):
        return self.get_work_in_release(release, team)
    
    def get_user_stories_for_sprint(self, sprint_id):
        '''
        Lists Ids of user stories assigned to a specified sprint
        @param sprint_id: GUS ID of sprint
        @return: array of gus work ids that represent stories in a sprint
        '''
        result = self.sf_session.query("Select Id from ADM_Work__c where RecordTypeId = '0129000000006gDAAQ' and Sprint__c = '{}'".format(sprint_id))
        return [x['Id'] for x in result['records']]
    
    def get_acceptance_criteria_for_work(self, work_id):
        '''
        Lists ids of acceptance criteria for a specified work item
        @param work_id: GUS ID of a work item
        @return: array of acceptance criteria ids
        '''
        result = self.sf_session.query("Select Id from ADM_Acceptance_Criterion__c where Work__c = '{}'".format(work_id))
        return [x['Id'] for x in result['records']]
    
    def get_gherkin_ac_records(self, ac_ids):
        '''
        Returns a filtered list of acceptance criteria where the description appears to be gherkin format
        @param ac_ids: an array of acceptance criteria ids to filter
        @return: a filtered array of acceptance criteria ids that appear to be in gherkin format
        '''
        soql = "Select Id, Description__c from ADM_Acceptance_Criterion__c where Id in ('{}')".format("','".join(ac_ids))
        result = self.sf_session.query(soql)
        out = []
        for x in result['records']:
            match = re.search("(Given .*)?When (.+)\nThen (.+)", x['Description__c'], re.IGNORECASE|re.DOTALL)
            if match:
                out.append(x['Id'])
                
        return out
    
    def get_specific_acceptance_criteria(self, work_id, acceptance_name):
        '''
        Retrieves the GUS ID of an acceptance criteria record that matches the given short name
        @param work_id: parent GUS ID to search within
        @param acceptance_name: name of acceptance criteria
        @return: the matching ID or None if not found
        @raise exception: if multiple acceptance criteria are found
        '''
        result = self.sf_session.query("Select Id from ADM_Acceptance_Criterion__c where Work__c = '{}' and Name = '{}'".format(work_id, acceptance_name))
        out = [x['Id'] for x in result['records']]
        if len(out) == 0:
            return None
        elif len(out) > 1:
            raise Exception("Duplicate Acceptance Criteria on story: {}".format(acceptance_name))
        else:
            return out[0]
        
    def update_acceptance_criteria_status(self, acceptance_id, status):
        '''
        Sets the status of a specified ac record to a specified status
        @param acceptance_id: GUS ID of ac record to update
        @param status: status value to set the record to
        '''
        self.sf_session.ADM_Acceptance_Criterion__c.update(acceptance_id, {'Status__c': status})
        
    def update_acceptance_criteria_details(self, acceptance_id, details):
        '''
        Updates the description of a specified acceptance criteria record
        @param acceptance_id: GUS ID of ac record to update
        @param details: New value for description
        '''
        self.sf_session.ADM_Acceptance_Criterion__c.update(acceptance_id, {'Description__c': details})
        
    def add_acceptance_criteria(self, work_id, acceptance_name, acceptance_details):
        '''
        Creates a new acceptance criteria record for the specified work id
        @param work_id: GUS ID of work item
        @param acceptance_name: Unique value of acceptance criteria name (short description)
        @param acceptance_details: Details of acceptance criteria, preferably in gherkin format
        '''
        result = self.sf_session.ADM_Acceptance_Criterion__c.create({'Work__c': work_id, 'Name': acceptance_name, 'Description__c': acceptance_details})
        return result['id']
    
    def get_user_stories_in_release(self, release, team=None, open_only=False):
        '''
        Finds all user story records that are targeted to any build in a release based on release names having a common
        prefix
        @param release: prefix common to all builds in the release
        @param team: ID of team to filter on if necessary
        @param open_only: True if you only wish list ids which are not closed
        @return: array of user story ids that are targeted to a release
        '''
        return self.get_work_in_release(release, team, open_only, stories_only=True)
        
    def get_work_in_release(self, release, team=None, open_only=False, stories_only=False):
        '''
        Finds all work records that are targeted to any build in a release based on release names having a common prefix
        @param release: prefix common to all builds in the release
        @param team: Team ID to filter on if necessary
        @param open_only: True if you only wish to list items that are open
        @param stories_only: True if you wish to limit list to user stories only
        @return: array of work ids that are targeted to a release
        '''
        build_soql = "Select Id from ADM_Build__c where Name like '{}%'".format(release)
        builds = self.sf_session.query(build_soql)
        if len(builds['records']) > 0:
            soql = "Select Id from ADM_Work__c where Scheduled_Build__c in ('{}')".format("','".join([x['Id'] for x in builds['records']]))
            if team is not None:
                soql += " and Scrum_Team__c = '{}'".format(team)
            if stories_only:
                soql += " and RecordTypeId = '{}'".format(self.WORK_RECORD_TYPES['User Story'])
            if open_only:
                soql += " and Closed__c = 0"
            result = self.sf_session.query(soql)
            return [x['Id'] for x in result['records']]
        else:
            return []
    
    def get_team_burndown_remaining(self, team):
        soql = "Select Id from ADM_Work__c where Scrum_Team__c = '{}' and Closed__c = 0 and RecordTypeId = '{}'".format(team, self.WORK_RECORD_TYPES['User Story'])
        result = self.sf_session.query(soql)
        work_ids = [x['Id'] for x in result['records']]
        return self.get_total_points_for_items(work_ids)
    
    def get_planned_velocity_for_release(self, release, team=None):
        work_ids = self.get_user_stories_in_release(release, team)
        return self.get_total_points_for_items(work_ids)
    
    def get_open_work_items(self, work_ids):
        soql = "Select Id from ADM_Work__c where Id in ('{}') and Closed__c = 0".format("','".join(work_ids))
        result = self.sf_session.query(soql)
        return [x['Id'] for x in result['records']]
    
    def get_closed_work_items(self, work_ids):
        soql = "Select Id from ADM_Work__c where Id in ('{}') and Closed__c = 1".format("','".join(work_ids))
        result = self.sf_session.query(soql)
        return [x['Id'] for x in result['records']]
    
    def get_work_items_with_acceptance_criteria(self, work_ids):
        soql = "Select Work__c, count(Id) c from ADM_Acceptance_Criterion__c where Work__c in ('{}') Group By Work__c".format("','".join(work_ids))
        result = self.sf_session.query(soql)
        return [x['Work__c'] for x in result['records']]
    
    def get_work_items_with_gherkin_acceptance_criteria(self, work_ids):
        soql = "Select Work__c, Description__c from ADM_Acceptance_Criterion__c where Work__c in ('{}')".format("','".join(work_ids))
        result = self.sf_session.query(soql)
        out = []
        for x in result['records']:
            if x['Description__c']:
                match = re.search("(Given .*)?When (.+)\nThen (.+)", x['Description__c'], re.IGNORECASE|re.DOTALL)
                if match and x['Work__c'] not in out:
                    out.append(x['Work__c'])
                
        return out
    
    def get_functional_format_stories(self, work_ids):
        soql = "Select Id, Details__c from ADM_Work__c where Id in ('{}')".format("','".join(work_ids))
        result = self.sf_session.query(soql)
        out = []
        for x in result['records']:
            if x['Details__c']:
                p = StoryParser(x['Details__c'])
                if p.is_functional():
                    out.append(x['Id'])
                else:
                    parents = self.get_parent_work_for_work(x['Id'])
                    good_parents = self.get_functional_format_stories(parents)
                    if len(good_parents) > 0:
                        out.append(x['Id'])
                
        return out
    
    def get_story_format_stories(self, work_ids):
        soql = "Select Id, Details__c from ADM_Work__c where Id in ('{}')".format("','".join(work_ids))
        result = self.sf_session.query(soql)
        out = []
        for x in result['records']:
            if x['Details__c']:
                p = StoryParser(x['Details__c'])
                if p.is_complete():
                    out.append(x['Id'])
                else:
                    parents = self.get_parent_work_for_work(x['Id'])
                    good_parents = self.get_story_format_stories(parents)
                    if len(good_parents) > 0:
                        out.append(x['Id'])
                
        return out
    
    def extract_personas_from_stories(self, stories):
        soql = "Select Id, Details__c from ADM_Work__c where Id in ('{}')".format("','".join(stories))
        result = self.sf_session.query(soql)
        out = {}
        for x in result['records']:
            if x['Details__c']:
                p = StoryParser(x['Details__c'])
                if p.is_functional():
                    persona = p.get_persona()
                    if persona not in out.keys():
                        out[persona] = 1
                    else:
                        out[persona] += 1
        return out        

    
    def get_total_points_for_items(self, work_ids):
        soql = "Select sum(Story_Points__c) Total from ADM_Work__c where Id in('{}')".format("','".join(work_ids))
        result = self.sf_session.query(soql)
        out = result['records'][0]['Total'] or 0
        return int(out)
    
    def get_average_age_of_work_items(self, work_ids):
        soql = "Select avg(Age__c) age from ADM_Work__c where ID in('{}')".format("','".join(work_ids))
        result = self.sf_session.query(soql)
        out = result['records'][0]['age'] or 0
        return int(out)
    
    def get_work_status_history(self, workid):
        result = self.sf_session.query("SELECT CreatedDate, OldValue, NewValue FROM ADM_Work__History where parentId = '{}' and Field = 'Status__c'".format(workid))
        out = []
        for x in result['records']:
            if x['CreatedDate'] not in [y[0] for y in out]:
                out.append((x['CreatedDate'], x['OldValue'], x['NewValue']))
                
        return out
    
    def get_themes_for_release(self, release, team=None):
        work = self.get_work_in_release(release, team=team)
        return self.get_themes_for_work(work)
    
    def get_themes_for_work(self, work):
        # split up the list
        soql = "Select Theme__c from ADM_Theme_Assignment__c where Work__c in('{}')".format("','".join(work))
        result = self.sf_session.query(soql)
        return [x['Theme__c'] for x in result['records']]
    
    def get_epics_for_work(self, work):
        soql = "Select distinct Epic__c from ADM_Work__c where Work__c in ('{}')".format("','".join(work))
        result = self.sf_session.query(soql)
        return [x['Epic__c'] for x in result['records']]
    
    def get_work_in_epic(self, epicid):
        soql = "Select Id from ADM_Work__c where Epic__c = '{}'".format(epicid)
        result = self.sf_session.query(soql)
        return [x['Id'] for x in result['records']]
    
    def filter_work_for_epic(self, work, epicid):
        soql = "Select Id from ADM_Work__c where Epic__c = '{epic}' and Id in ('{work}')".format(epic=epicid, work="','".join(work))
        result = self.sf_session.query(soql)
        return [x['Id'] for x in result['records']]
