#!/usr/bin/env python
# coding: utf-8
#
# <h - an(other) simple static site generator!>
# Copyright (C) <2013-2014>  Jean Pimentel <contato@jeanpimentel.com.br>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.

import argparse
import sys
import os
import codecs
import re
import glob
from datetime import datetime
import shutil
import locale
import warnings
from jinja2 import Environment, FileSystemLoader
import markdown
import yaml
import distutils.dir_util

def main():

    args = get_args()

    if not is_valid_source_dir(args['source']):
        sys.exit('Error: Invalid source directory')

    configs = get_configs(args)

    if 'locale' in configs:
        try:
            locale.setlocale(locale.LC_ALL, configs['locale'])
        except Exception:
            warnings.warn('Unsupported locale setting', RuntimeWarning)

    view = Environment(loader=FileSystemLoader(os.path.join(configs['source'], 'templates')))
    view.globals['configs'] = configs
    view.filters['date'] = format_date

    generate(configs, view)


def get_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("-s", "--source", help="source directory")
    parser.add_argument("-d", "--destination", help="destination directory")
    parser.add_argument("-c", "--config", default='config.yml', help="config filename")
    parser.add_argument("-v", "--version", action='version', version=get_version())
    args = parser.parse_args()
    if not args.source:
        sys.exit('Error: You must specify source directory')
    if not args.destination:
        sys.exit('Error: You must specify destination directory')
    result = {
        'source': os.path.realpath(args.source),
        'destination': os.path.realpath(args.destination),
        'config': os.path.join(os.path.realpath(args.source), args.config)
    }
    return result


def get_configs(args):
    configs = {
        'per_page': 10,
        'source': args['source'],
        'destination': args['destination'],
    }
    if os.path.isfile(args['config']):
        with open(args['config'], 'r') as f:
            configs.update(yaml.load(f.read()))
    return configs


def get_version():
    import pkg_resources
    return pkg_resources.require('h')[0].version


def generate(configs, view):

    shutil.rmtree(configs['destination'], True)
    os.makedirs(configs['destination'])

    distutils.dir_util.copy_tree(os.path.join(configs['source'], 'public'), configs['destination'])

    info = {}
    info['posts'] = sorted(posts(os.path.join(configs['source'], 'posts')), key=lambda p: p['date'], reverse=True)
    info['pages'] = sorted(pages(os.path.join(configs['source'], 'pages')), key=lambda p: p['order'])
    info['paginator'] = list(enumerate(paginate(info['posts'], configs['per_page']), start=1))
    info['tags'] = tags(info['posts'])

    view.globals['info'] = info

    render_posts(info['posts'], configs, view)
    render_pages(info['pages'], configs, view)
    render_paginator(info['paginator'], configs, view)
    render_tags(info['tags'], configs, view)


def posts(directory):
    posts = []
    for filename in sorted(os.listdir(directory)):
        with codecs.open(os.path.join(directory, filename), 'r', 'utf-8') as f:
            post = parse_post(filename, f.read())
            if post:
                posts.append(post)
    return posts


def pages(directory):
    pages = []
    for filename in sorted(os.listdir(directory)):
        with codecs.open(os.path.join(directory, filename), 'r', 'utf-8') as f:
            page = parse_page(filename, f.read())
            if page:
                pages.append(page)
    return pages


def tags(posts):
    tags = {}
    for post in posts:
        if post['tags']:
            for name, slug in post['tags']:
                if slug not in tags:
                    tags[slug] = {'slug': slug, 'name': name, 'posts': []}
                tags[slug]['posts'].append(post)
    return tags


def parse_post(filename, content):
    metadata, content = extract_metadata(content)
    if not metadata:
        return None
    post = yaml.load(metadata)
    if 'tags' in post.keys():
        for i, tag in enumerate(post['tags']):
            name, slug = tag, tag.lower().replace(' ', '-')
            if '|' in tag:
                name, slug = tag.split('|')
            post['tags'][i] = (name, slug)
    post['date'] = filename[:10]
    post['slug'] = filename[11:].split('.md')[0]
    md = markdown.Markdown(output='html5', extensions=['extra'])
    post['content'] = md.convert(content)
    if not 'excerpt' in post.keys():
        paragraphs = re.search(r'<p>(.*?)<\/p>', post['content'])
        if paragraphs is not None:
            post['excerpt'] = re.compile(r'<.*?>').sub('', paragraphs.groups()[0])
            post['content'] = post['content'][paragraphs.end():]
    return post


def parse_page(filename, content):
    metadata, content = extract_metadata(content)
    if not metadata:
        return None
    page = yaml.load(metadata)
    page['slug'] = filename.split('.md')[0]
    md = markdown.Markdown(output='html5', extensions=['extra'])
    page['content'] = md.convert(content)
    return page


def extract_metadata(content):
    lines = content.splitlines()
    dashes = []
    for i, line in enumerate(lines):
        if line.startswith('---'):
            dashes.append(i)
        if len(dashes) == 2:
            return '\n'.join(lines[dashes[0]+1:dashes[1]]), '\n'.join(lines[dashes[1]+1:])
    return None, content


def paginate(data, per_page):
    for start in iter(range(0, len(data), per_page)):
        yield data[start:start + per_page]


def render_posts(posts, configs, view):
    template = view.get_template('post.html')
    for post in posts:
        path = os.path.join(configs['destination'], post['slug'] + '.html')
        write(path, template.render(post=post).encode('utf-8'))


def render_pages(pages, configs, view):
    template = view.get_template('page.html')
    for page in pages:
        path = os.path.join(configs['destination'], page['slug'] + '.html')
        write(path, template.render(page=page).encode('utf-8'))


def render_paginator(paginator, configs, view):
    template = view.get_template('posts.html')
    for page, items in paginator:
        path = os.path.join(configs['destination'], str(page) + '.html')
        write(path, template.render(page=page, size=len(paginator), posts=items).encode('utf-8'))
    shutil.copyfile(os.path.join(configs['destination'], '1.html'), os.path.join(configs['destination'], 'index.html'))


def render_tags(tags, configs, view):
    template = view.get_template('tag.html')
    for tag, info in tags.items():
        path = os.path.join(configs['destination'], tag + '.html')
        write(path, template.render(tag=info).encode('utf-8'))


def write(filename, content):
    with open(filename, 'wb') as f:
        f.write(content)


def is_valid_source_dir(source):
    return \
        os.path.isdir(source) and \
        os.path.isdir(os.path.join(source, 'posts')) and \
        os.path.isdir(os.path.join(source, 'pages')) and \
        os.path.isdir(os.path.join(source, 'templates')) and \
        os.path.isdir(os.path.join(source, 'public'))


def format_date(value, dateformat):
    return datetime.strptime(value, '%Y-%m-%d').strftime(dateformat)


if __name__ == "__main__":
    main()
