# Copyright (c) 2009-2011 Te-je Rodgers
#
# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation
# files (the "Software"), to deal in the Software without
# restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or
# sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following
# conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.
#

import abc
import itertools


class Finder(object):
    """
    An abstract base class for Plugin finders...
    """

    __metaclass__ = abc.ABCMeta

    @abc.abstractmethod
    def __iter__(self):
        """
        Iterate through all of the discovered plugins
        """

    def find(self, hook):
        """Return all the plugins that use the specified hook."""
        return itertools.ifilter(lambda p: p.implements(hook), self)

class Plugin(object):
    """
    An abstract base class for a plugin
    """

    __metaclass__ = abc.ABCMeta

    @abc.abstractmethod
    def implements(self, hook):
        """
        Return whether the the plugin has implementers for the
        given hook.
        """

    @abc.abstractmethod
    def implementers(self, hook):
        """
        Return an iterator to the extensions that implement the
        given hook.
        """

    @abc.abstractmethod
    def resourcestream(self, path):
        """
        Return an open file (or file-like) object for the resource
        given by path.
        """


class CachedFinder(Finder):

    def __init__(self, finder):
        self.finder = finder
        self.cache = None

    def __iter__(self):
        if self.cache is None:
            self.cache = []
            for plugin in self.finder:
                self.cache.append(plugin)
                yield plugin

        else:
            for plugin in self.cache:
                yield plugin


class ChainedFinder(Finder):

    def __init__(self, *finders):
        self.finders = finders

    def __iter__(self):
        return itertools.chain(*self.finders)


global_finders = set()
def register_global_finder(finder):
    global_finders.add(finder)

def unregister_global_finder(finder):
    global_finders.remove(finder)

def find_globally():
    return ChainedFinder(*global_finders)
