# Copyright (c) 2009-2011 Te-je Rodgers
#
# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation
# files (the "Software"), to deal in the Software without
# restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or
# sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following
# conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.
#

import itertools

from .finder import find_globally
from .internal import InternalPlugin


class Interface(object):
    """
    An interface is one way to extend an application using hatch
    """

    finder = None

    @classmethod
    def fqn(cls):
        """Return the fully-qualified name of the interface."""
        return "{0.__module__}.{0.__name__}".format(cls)

    @classmethod
    def hook(cls):
        """
        Return a string that uniquely identifies the interface.
        This must be unique across the hatch instance; a hook cannot
        be shared across multiple extension types.
        """
        # By default, use the interface's fully qualified name.
        # An interface can reimplement the class method to provide
        # a more suitable hook name if necessary.
        return cls.fqn()

    @classmethod
    def plugins(cls):
        if cls.finder is None:
            finder = find_globally()
        else:
            finder = cls.finder

        return finder.find(cls.hook())

    @classmethod
    def implementers(cls):
        return itertools.chain(
            *[plugin.implementers(cls.hook()) for plugin in cls.plugins()])

    @classmethod
    def instances(cls, *args, **kwargs):
        for imp in cls.implementers():
            yield imp(*args, **kwargs)

    @classmethod
    def apply(cls, funcname, *args, **kwargs):
        results = []
        for inst in cls.instances():
            try:
                results.append(getattr(inst, funcname)(*args, **kwargs))
            except Exception as e:
                results.append(e)
        else:
            return results

    @classmethod
    def first(cls, *args, **kwargs):
        """
        Return an instance of the 'first' implementation of the interface.

        What this method actually does is return
        the first external implementation of the interface, falling
        back to internal implementations only when no external
        implementations are found.

        That means that if an internal implementation is found first,
        it will be ignored in favor of the first external
        implementation, and returned only if no external implementations
        are found.

        This method is useful for applications that want to use a
        single external component to perform a particular task, but
        provides its own implementation to fall back on.

        The *args and **kwargs are passed directly into the
        implementation's constructor to retrieve the instance returned.

        If no implementations are found, LookupError is raised.

        """
        for plugin in cls.plugins():
            if plugin is InternalPlugin:
                continue
            else:
                p = plugin
                break
        else:
            p = InternalPlugin

        # Verify that the plugin has an implementation
        if not p.implements(cls.hook()):
            raise LookupError("There are no implementations of this interface")
        else:
            return next(p.implementers(cls.hook()))(*args, **kwargs)
