# Copyright (c) 2009-2011 Te-je Rodgers
#
# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation
# files (the "Software"), to deal in the Software without
# restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or
# sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following
# conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.
#


from .finder import find_globally
from .utils import iter_extensions

class Filters(object):
    """
    An extension pattern that progressively passes data extensions,
    allowing them to alter it, returning the final altered data
    to the application.
    """

    def __init__(self, hook, finder=None, check_func=None):
        self.hook = hook
        self.finder = finder

        if check_func is None:
            self.check = lambda data: data
        elif callable(check_func):
            self.check = check_func
        else:
            raise ValueError("Check function must be callable")

    def __repr__(self):
        return u"Filters<{0.hook}>".format(self)

    def __iter__(self):
        """Iterate through all the filter functions."""
        finder = self.finder or find_globally()
        return iter_extensions(finder, self.hook)

    def __call__(self, data):
        """Pass the data through the filters."""
        for fun in self:
            new_data = fun(data)

            if self.check(new_data) is None:
                break

            data = new_data

        return data

    def filter(self, data):
        """Pass the data through the filters."""
        # Nothing but a shallow wrapper
        return self(data)

