# Copyright (c) 2009-2011 Te-je Rodgers
#
# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation
# files (the "Software"), to deal in the Software without
# restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or
# sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following
# conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.
#

try:
    import futures
except ImportError:
    FUTURES = False
else:
    FUTURES = True

from .finder import find_globally
from .utils import iter_extensions

class Observers(object):

    def __init__(self, hook, finder=None, concurrency='thread'):
        self.hook = hook
        self.finder = finder

        if concurrency == 'thread':
            self.executor = futures.ThreadPoolExecutor(4) if FUTURES else None
        elif concurrency == 'process':
            self.executor = futures.ProcessPoolExecutor() if FUTURES else None
        elif concurrency is None:
            self.executor = None
        else:
            raise ValueError("Invalid concurrency mode")

    def __repr__(self):
        return u"Observers<{0.hook}>".format(self)

    def __del__(self):
        self.executor.shutdown(False)

    def __iter__(self):
        """Iterate through all the observers."""
        finder = self.finder or find_globally()
        return iter_extensions(finder, self.hook)

    def __call__(self, *args, **kwargs):
        """Alert all of the observers with the given arguments."""
        for observer in self:
            if self.executor is not None:
                self.executor.submit(observer, *args, **kwargs)
            else:
                observer(*args, **kwargs)

    def notify(self):
        """Notify the observers that the event has occured"""
        # Nothing but a shallow wrapper
        self()
