# Copyright (c) 2009-2011 Te-je Rodgers
#
# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation
# files (the "Software"), to deal in the Software without
# restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or
# sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following
# conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.
#

import functools
import itertools


class SingletonMeta(type):
    objs = {}
    def __call__(cls, *args, **kwargs):
        obj = SingletonMeta.objs.get(cls)
        if obj is None:
            obj = type.__call__(cls, *args, **kwargs)
            SingletonMeta.objs[cls] = obj
        return obj


def unique(fun):
    """A decorator for functions that should return an iterator of unique items."""
    @functools.wraps(fun)
    def iter_unique(*args, **kwargs):
        seen = []
        for item in fun(*args, **kwargs):
            if item not in seen:
                yield item
                seen.append(item)
    return iter_unique

def iter_extensions(finder, hook):
    """
    Return an iterator to all the implementers of the given hook.

    ``finder`` is a Finder instance that can be used to find plugins.
    ``hook`` is the name of the hook that the implementer is attached to.

    This function is similar to finder.find, except that finder.find
    returns all *plugins* that implement the hook (each plugin returned
    contains one or more implementer), while this function returns
    the *implementers* themselves. It wraps finder.find
    """
    return itertools.chain(
        *[plugin.implementers(hook) for plugin in finder.find(hook)])
