# -*- encoding: iso-8859-15 -*-

#################################################################
# haufe.hrs.configuration - a pseudo-hierarchical configuration
# management infrastructure
# (C) 2008, Haufe Mediengruppe, Freiburg
#################################################################


"""
Tests for haufe.hrs.configuration module
"""

import os
import unittest
from unittest import TestCase, TestSuite, makeSuite
from haufe.hrs.configuration import generateParser, OptionLookup

dirname = os.path.dirname(os.path.abspath(__file__))

class ConfigurationTests(TestCase):

    def _generateParser(self, name):
        return generateParser(os.path.join(dirname, name))

    def testParseModel(self):
        parser = self._generateParser('model') 

    def testParseNonExistingModelDirectory(self):
        self.assertRaises(ValueError, generateParser, 'non-existing-model-dir')

    def testOptionLookupWithDefaults(self):
        parser = self._generateParser('model') 
        parser.add_file(os.path.join(dirname, 'empty-config.ini'))
        opts = parser.parse()
        OL = OptionLookup(opts)
        self.assertEqual(OL.get('cms.path'), '/foo/bar') 
        self.assertEqual(OL.get('cms.checkout_path'), None)
        self.assertEqual(OL.get('cms.port'), 42)
        self.assertEqual(OL.get('cms.databases.db1'), None)
        self.assertEqual(OL.get('cms.databases.db2'), None)

    def testOptionLookupWithConfigurationFile(self):
        parser = self._generateParser('model') 
        parser.add_file(os.path.join(dirname, 'example-config.ini'))
        opts = parser.parse()
        OL = OptionLookup(opts)
        self.assertEqual(OL.get('cms.path'), '/a/b/c') 
        self.assertEqual(OL.get('cms.checkout_path'), '/foo/bar')
        self.assertEqual(OL.get('cms.port'), 777)
        self.assertEqual(OL.get('cms.databases.db1'), 'n/a')
        self.assertEqual(OL.get('cms.databases.db2'), None)

    def testOptionLookupWithConfigurationFileWithPrefix(self):
        parser = self._generateParser('model') 
        parser.add_file(os.path.join(dirname, 'example-config.ini'))
        opts = parser.parse()
        OL = OptionLookup(opts, 'cms')
        self.assertEqual(OL.get('path'), '/a/b/c') 
        self.assertEqual(OL.get('checkout_path'), '/foo/bar')
        self.assertEqual(OL.get('port'), 777)
        self.assertEqual(OL.get('databases.db1'), 'n/a')
        self.assertEqual(OL.get('databases.db2'), None)
        self.assertEqual(OL.has('databases.db2'), True)
        self.assertEqual(OL.has('gibt.es.nicht'), False)

    def testOptionLookupWithConfigurationFileWithPrefixAndFullName(self):
        parser = self._generateParser('model') 
        parser.add_file(os.path.join(dirname, 'example-config.ini'))
        opts = parser.parse()
        OL = OptionLookup(opts, 'cms')
        self.assertEqual(OL.get('cms.path'), '/a/b/c') 
        self.assertEqual(OL.get('cms.checkout_path'), '/foo/bar')
        self.assertEqual(OL.get('cms.port'), 777)
        self.assertEqual(OL.get('cms.databases.db1'), 'n/a')
        self.assertEqual(OL.get('cms.databases.db2'), None)
        self.assertEqual(OL.has('cms.databases.db2'), True)
        self.assertEqual(OL.has('cms.gibt.es.nicht'), False)


    def testOptionLookupWithNonExistingKeys(self):
        parser = self._generateParser('model') 
        parser.add_file(os.path.join(dirname, 'example-config.ini'))
        opts = parser.parse()
        OL = OptionLookup(opts)
        self.assertRaises(ValueError, OL.get,'gibt.es.nicht')


def test_suite():
    return TestSuite((
        makeSuite(ConfigurationTests),
        ))

if __name__ == '__main__':
    unittest.main()


