import select
import uuid
import functools
from qpid import messaging
from qpid.messaging.exceptions import LinkClosed, Empty

import minsol
import hens


class Server(minsol.Object):
    request_class = hens.Request
    response_class = hens.Response

    def init(self, acceptor_class, connection, queue_name, **other):
        self.acceptor_class = acceptor_class
        self.connection = connection
        self.queue_name = queue_name

    def start(self):
        self.init_broker()
        self.listen()

    def stop(self):
        self.receiver.close()

    def init_broker(self):
        if not self.connection.opened():
            self.connection.open()

        session = self.connection.session("rpc init")

        q = "%s; {"\
            "create:always, " \
            "node: {durable:False, x-declare:{auto-delete:True}}"\
        "}"
        q = q % self.queue_name
        session.sender(q)

        session.close()

    def listen(self):
        session = self.connection.session("rpc listen")
        self.receiver = session.receiver(self.queue_name)

        def _executed(reply_to, request, result):
            if not reply_to:
                return

            response = self.response_class(request=request, result=result)

            raw_response = response.as_raw()
            tmp_s = session.sender(reply_to)
            tmp_s.send(raw_response)
            tmp_s.close()

        while True:
            try:
                msg = self.receiver.fetch()

            except (LinkClosed, KeyboardInterrupt, select.error):
                break

            session.acknowledge(msg)

            raw_request = msg.content
            acceptor = self.acceptor_class()
            request = None

            try:
                request = self.request_class.get_from_raw(raw_request)

                callback = functools.partial(_executed, msg.reply_to, request)
                request.execute(acceptor, callback)

            except hens.Error as error:
                _executed(msg.reply_to, request, error)

        session.close()


class Client(hens.Client):
    def init(self, connection, queue_name, timeout=None, **kwargs):
        self.connection = connection
        if not self.connection.opened():
            self.connection.open()

        self.queue_name = queue_name
        self.timeout = timeout

    def _fetch_call(self, raw_request):
        session = self.connection.session("rpc call")

        response_queue_name = self.gen_response_queue_name()
        response_receiver = session.receiver(response_queue_name)

        sender = session.sender(self.queue_name)

        raw_request = messaging.Message(raw_request,
            reply_to=response_queue_name)

        sender.send(raw_request)

        try:
            msg = response_receiver.fetch(timeout=self.timeout)

        except Empty:
            raise hens.InvalidResponse()

        except (LinkClosed, KeyboardInterrupt, select.error):
            raise LinkClosed()

        raw_response = msg.content

        session.close()
        return raw_response

    def gen_response_queue_name(self):
        tmp_ident = uuid.uuid4().hex

        q = "%s.tmp-%s; {"\
            "create:always, " \
            "node: {"\
                "durable:False, "\
                "x-declare:{auto-delete:True, exclusive:True}}}"

        q = q % (self.queue_name, tmp_ident)
        return q


class JSONServer(Server):
    request_class = hens.JSONRequest
    response_class = hens.JSONResponse


class JSONClient(Client, hens.JSONClient):
    pass









