import jsonpickle
import functools

import minsol


class Acceptor(minsol.Object):
    """
    .. note::
        first argument ``callback`` should be ignored, if your
        client is not asynchronous.
    """

    def get_method(self, name):
        try:
            method = getattr(self, name)
            a = method.is_rpc_method
            del a

        except AttributeError:
            raise MethodNotFound()

        return method


class Request(minsol.Object):
    def init(self, ident, method_name, args, kwargs, **other):
        self.id = ident
        self.method_name = method_name
        self.args = args
        self.kwargs = kwargs

    def execute(self, acceptor, callback):
        method = acceptor.get_method(self.method_name)

        try:
            method(callback, *self.args, **self.kwargs)

        except TypeError as error:
            raise InvalidParams(data=error.message)

    def as_dict(self):
        raise NotImplementedError

    def as_raw(self):
        raise NotImplementedError

    @classmethod
    def get_dict_from_raw(cls, raw_data):
        raise NotImplementedError

    @classmethod
    def get_from_raw(cls, raw_data):
        raise NotImplementedError


class Response(minsol.Object):
    def init(self, request, ident=None, result=None, **kwargs):
        self.request = request
        self.raw_id = ident

        self.result = result

        if self.raw_id and self.request.id != self.raw_id:
            raise InvalidResponse()

    def as_dict(self):
        raise NotImplementedError

    def as_raw(self):
        raise NotImplementedError

    @classmethod
    def get_from_raw(cls, raw_data):
        raise NotImplementedError

    @classmethod
    def get_dict_from_raw(cls, request, raw_data):
        raise NotImplementedError


class JSONRequest(Request):
    possible_keys = ["id", "method", "args", "kwargs"]

    def as_dict(self):
        request = {
            "id": self.id,
            "method": self.method_name
        }

        if self.args:
            request["args"] = self.args

        if self.kwargs:
            request["kwargs"] = self.kwargs

        return request

    def as_raw(self):
        request = self.as_dict()
        request = jsonpickle.encode(request)
        return request

    @classmethod
    def get_dict_from_raw(cls, raw_data):
        try:
            query = str(raw_data)
            request = jsonpickle.decode(query)

        except ValueError as error:
            raise ParseError(data=error.message)

        cls.validate_data(request)

        return request

    @classmethod
    def validate_data(cls, data):

        if data.__class__ != dict:
            raise InvalidRequest("Request MUST be dict!")

        for k in data.keys():
            if k not in cls.possible_keys:
                raise InvalidRequest("Bad key %s" % k)

        if "method" not in data.keys():
            raise InvalidRequest("key 'method' MUST be specified!")

        if "args" in data.keys():
            if data["args"].__class__ != list:
                raise InvalidRequest("key 'args' MUST be list!")

        if "kwargs" in data.keys():
            if data["kwargs"].__class__ != dict:
                raise InvalidRequest("key 'kwargs' MUST be dict!")

    @classmethod
    def get_from_raw(cls, raw_data):
        request = cls.get_dict_from_raw(raw_data)

        request_id = request.get("id", None)
        request = cls(
            ident=request_id, method_name=request["method"],
            args=request.get("args", []),
            kwargs=request.get("kwargs", {}))

        return request


class JSONResponse(Response):
    possible_keys = ["id", "result"]

    def as_dict(self):
        request_id = self.request.id if self.request else None

        response = {
            "id": request_id,
            "result": self.result
        }

        return response

    def as_raw(self):
        response = self.as_dict()

        response = jsonpickle.encode(response)
        return response

    @classmethod
    def get_dict_from_raw(cls, request, raw_data):
        try:
            raw_data = str(raw_data)
            response = jsonpickle.decode(raw_data)

        except ValueError as error:
            raise ParseError(data=error.message)

        cls.validate_data(response)
        return response

    @classmethod
    def get_from_raw(cls, request, raw_data):
        response = cls.get_dict_from_raw(request, raw_data)

        response = cls(request=request, **response)
        return response

    @classmethod
    def validate_data(cls, data):
        if data.__class__ != dict:
            raise InvalidRequest("Response MUST be dict!")

        for k in data.keys():
            if k not in cls.possible_keys:
                raise InvalidRequest("Bad key %s" % k)

        if "result" not in data.keys():
            raise InvalidRequest("key 'result' MUST be specified!")


class Client(minsol.Object):
    request_class = Request
    response_class = Response

    def init(self, **kwargs):
        self.last_id = 1

    def __getattr__(self, name):
        if name[:2] == "__":
            raise AttributeError

        return functools.partial(self._rpc_call, name)

    def _rpc_call(self, method_name, *args, **kwargs):
        args = list(args)

        request = self.request_class(
            ident=self.last_id, method_name=method_name,
            args=args, kwargs=kwargs)

        self.last_id += 1

        raw_request = request.as_raw()
        raw_response = self._fetch_call(raw_request)

        response = self.response_class.get_from_raw(request, raw_response)

        result = response.result
        if isinstance(result, Exception):
            raise result

        return result

    def _fetch_call(self, raw_request):
        """
        @return: raw_response
        """
        raise NotImplementedError


class JSONClient(Client):
    request_class = JSONRequest
    response_class = JSONResponse


class Error(minsol.Error):
    code = -32603
    message = 'Internal error'

    def __init__(self, data=None, request_id=None):
        self.data = data
        self.request_id = request_id


class ParseError(Error):
    code = -32700
    message = 'Parse error'


class InvalidRequest(Error):
    code = -32600
    message = 'Invalid request'


class MethodNotFound(Error):
    code = -32601
    message = 'Method not found'


class InvalidParams(Error):
    code = -32602
    message = 'Invalid params'


class InvalidResponse(Error):
    code = -32604
    message = 'Invalid response'


class ServerError(Error):
    code = -32000
    message = 'Server error'


def method(func):
    func.is_rpc_method = True
    return func


