'''Copy one or more files from one mercurial repository to another, preserving
history.
'''

import argparse, json, os, subprocess, sys, tempfile
import shutil



class NonExistentFile(Exception):
    '''A required file in the source repo does not exist.'''
    def __init__(self, path):
        super(NonExistentFile, self).__init__(
            'The file "%s" does not exist' % path)


class DestinationFileExists(Exception):
    '''A file in the dest repo does already exists.'''
    def __init__(self, path):
        super(DestinationFileExists, self).__init__(
            'The file "%s" already exists' % path)


class InvalidRepositoryError(Exception):
    '''A file in the dest repo does already exists.'''
    def __init__(self, path):
        super(InvalidRepositoryError, self).__init__(
            'The path "%s" is not contained in a Mercurial repository.' % path)


def strip_prefix(path, prefix):
    '''Remove a prefix from a path.  
    
    If `path` starts with `prefix`, `prefix` is removed.  Also strip any 
    leading '/' characters. 
    
    :param path: the path to remove the prefix from
    :type path: str
    :param prefix: the prefix to remove from the path
    :type prefix: str
    :return: `path` with `prefix` removed.
    '''
    if path.startswith(prefix):
        path = path[len(prefix):]
    if path.startswith('/'):
        path = path[1:]
    return path


def add_prefix(path, prefix):
    '''Add a prefix to a path.  
    
    :param path: the path to prepend the `prefix` to
    :type path: str
    :param prefix: the prefix to prepend to `path`
    :type prefix: str
    :return: `prefix`/`path` 
    '''
    if prefix is None:
        return path
    return os.path.join(prefix, path)


def rename(path, strip_prefixes, output_prefix):
    '''Strip a list of prefixes from a path, and add an output prefix.  
    
    :param path: the path to modify
    :type path: str
    :param strip_prefixes: a list of prefixes to try and strip from `path`
    :type strip_prefixes: `list[str]|None`
    :param prefix: the prefix to prepend to `path`
    :type prefix: `str|None`
    :return: the modified path
    '''
    if strip_prefixes is None:
        strip_prefixes = list()
    path = path.strip('/')
    result = path
    for p in strip_prefixes:
        result = strip_prefix(result, p)
        if result != path:
            break
    return add_prefix(result, output_prefix)


def cd_exec(path, func, *args, **kwargs):
    '''Change the current working dir, execute a function, then restore the 
    working directory to its original path.
    
    :param path: the path to change the cwd to
    :type path: str
    :param func: the function to execute
    :type func: a function or any callable
    :param *args: the arguments to pass to `func` 
    :param *kwargs: the keyword arguments to pass to `func` 
    :return: the return value of `func`  
    '''
    old_dir = os.getcwd()
    os.chdir(path)
    try:
        return func(*args, **kwargs)
    finally:
        os.chdir(old_dir)


def hg_cmd(*args, **kwargs):
    '''Execute a mercurial command.
    
    The command executed is:
        hg arg[0] arg[1] ...
    
    :param *args: arguments to pass to the 'hg' command
    :type prefix: tuple[str]
    :param pretend: if `True` just print out what would be executed.
    :type pretend: bool
    :return: output of the command
    :rtype: str
    :raises: `subprocess.CalledProcessError`
    '''
    pretend = kwargs.pop('pretend', False)
    cmd = ('hg',) + args
    if pretend:
        print("Executing command:")
        print("    " + str(cmd))
        return ''
    output = subprocess.check_output(cmd).strip() # pylint: disable=E1103
    return output


def rep_cmd(repo_path, *args, **kwargs):
    return cd_exec(repo_path, hg_cmd, *args, **kwargs)


def hg_log(repo_path, *args):
    style = r"""file     = '"{file|escape}",'

changeset = '\\{   "revision": "{rev}",\n    "node": "{node}",\n    "branches": [{branches}],\n    "bookmarks": [{bookmarks}],\n    "tags":[{tags}],\n    "parents":[{parents}],\n    "email": "{author|email|xmlescape}",\n    "name": "{author|person|xmlescape}",\n    "date": "{date|rfc3339date}",\n    "desc": "{desc|xmlescape}",\n    "file_adds": [{file_adds}],\n    "file_dels": [{file_dels}],\n    "file_mods": [{file_mods}],\n    "file_copies": [{file_copies}] },\n'

file_add  = '"{file_add|escape}", '
last_file_add='"{file_add|escape}"'
file_mod  = '"{file_mod|escape}", '
last_file_mod='"{file_mod|escape}"'
file_del  = '"{file_del|escape}", '
last_file_del='"{file_del|escape}"'

file_copy = '["{source|escape}", "{name|escape}"],'
last_file_copy = '["{source|escape}", "{name|escape}"]'

parent = '\{"revision": "{rev}", "node": "{node}" },\n'
last_parent = '\{"revision": "{rev}", "node": "{node}" }\n'
branch = '"{branch|escape}",\n'
last_branch = '"{branch|escape}"\n'
tag = '"{tag|escape}",\n'
last_tag = '"{tag|escape}"'
bookmark = '"{bookmark|escape}",\n'
last_bookmark = '"{bookmark|escape}"'
extra = '"{key|escape}": "{value|escape}",\n'
last_extra = '"{key|escape}": "{value|escape}",'
"""
    fd, p = tempfile.mkstemp('.style', 'hgcp-json.', text=True)
    os.write(fd, style)
    os.close(fd)
    output = rep_cmd(repo_path, 'log', '--style', p)
    output = '[' + output[:-1] + ']'
    os.remove(p)
    log = json.loads(output)
    result = []
    for entry in log:
        entry['revision'] = int(entry['revision'])
        for e in entry['parents']:
            if 'revision' in e:
                e['revision'] = int(e['revision'])
        result.append(entry)
    return result


def rep_has_changes(path):
    '''Determine whether a repository has uncommitted changes.
    
    :param path: the path of the repository root
    :type path: `str` 
    :return: `True` if the repository at `path` has uncommitted changes.
    :rtype: `bool`
    '''
    changes = rep_cmd(path, 'status', '-q', pretend=False)
    return (len(changes) > 0)


def get_repository(path):
    '''Determine the root directory of the repository that contains `path`.
    
    :param path: the path
    :type files: str
    :return: The root directory of the repository that contains `path`
    :rtype: `str`
    '''
    if not os.path.exists(path):
        raise NonExistentFile(path)

    if os.path.isfile(path):
        path = os.path.dirname(path)
    try:
        return rep_cmd(path, 'root').strip()
    except subprocess.CalledProcessError:
        raise InvalidRepositoryError(path)


def cross_repo_copy(files,
                    dest_dir,
                    strip_prefixes=None,
                    pretend=False,
                    action='copy'):
    '''Copy (or move) files from one repository to another.
    
    :param files: A list of files to copy (or move).  The files must all be 
      contained in the same mercurial repository.
    :type files: list[str]
    :param dest_dir: The path to the destination directory.  The directory 
      must exist, and must be inside a mercurial repository.
    :type dest_dir: str
    :param strip_prefixes: a list of prefixes to strip from `files` when 
       determining their location in the destination repository.
    :type strip_prefixes: list[str]
    :param pretend: if `True` just print out what would be executed.
    :type pretend: bool
    :param action: which action to perform: copy or move
    :type move: str `'copy'` or `'move'`
    '''
    if not os.path.exists(dest_dir):
        print('Dest path does not exist.')
        return False
    result = True
    sroot = set([get_repository(f) for f in files])
    if len(sroot) == 0:
        print("Error: the specified files are not contained in a mercurial" +
              " repository.")
        return False
    if len(sroot) > 1:
        print("Error: the specified files exist in more than one mercurial" +
              " repository")
        return False
    sroot = sroot.pop()

    droot = get_repository(dest_dir)

    # ensure that there are no changes since the last commit in both
    # the source and dest repositories
    sc = rep_has_changes(sroot)
    if sc:
        print("Error: source repository has uncommitted changes.")
        return False
    dc = rep_has_changes(droot)
    if dc:
        print("Error: destination repository has uncommitted changes.")
        return False
    strip_prefixes = list() if strip_prefixes is None else strip_prefixes
    rdest = strip_prefix(os.path.abspath(dest_dir), droot)
    # fmap is a sequence of (source, dest) tuples
    files = [strip_prefix(os.path.abspath(f), sroot) for f in files]

    fmap = [(f, rename(f, strip_prefixes, rdest)) for f in files]

    for s, d in fmap:
        sp = os.path.join(sroot, s)
        dp = os.path.join(droot, d)
        # ensure that the source file exists
        if not os.path.exists(sp):
            raise NonExistentFile(sp)
        # ensure that the dest file does not exist
        if os.path.exists(dp):
            raise DestinationFileExists(dp)

    filemap = ['include "%s"' % f for f in files]
    filemap += ['rename "%s" "%s"' % (s, d) for (s, d) in fmap if s != d]
    temp_repo_dir = tempfile.mkdtemp(prefix='hgcp.')
    cd_exec(temp_repo_dir, hg_cmd, 'init', pretend=pretend)

    fd, filemap_path = tempfile.mkstemp(prefix='hgcp.filemap.',
                                        suffix='.txt',
                                        text=True)
    os.write(fd, '\n'.join(filemap))
    os.close(fd)

    if pretend:
        print("filemap:")
        print('    ' + ('\n    '.join(filemap)))

    # import desired files to temporary repository
    hg_cmd('convert', '--filemap', filemap_path,
            sroot, temp_repo_dir, pretend=pretend)
    cd_exec(temp_repo_dir, hg_cmd, 'update')

    cd_exec(droot, hg_cmd, 'pull', '--force', temp_repo_dir, pretend=pretend)
    try:
        cd_exec(droot, hg_cmd, 'merge', pretend=pretend)
    except subprocess.CalledProcessError:
        cd_exec(droot, hg_cmd, 'update', pretend=pretend)

    if action == 'move':
        cd_exec(sroot, hg_cmd, 'remove', *files, pretend=pretend)
    # remove the temporary filemap
    os.remove(filemap_path)
    # remove the temporary repository
    shutil.rmtree(temp_repo_dir)
    return result


EPILOG = """
The source and destination repositories must not have any uncommitted changes.
The changes made to the ${src_and}destination repositories are not committed.

In the following examples, there are two repositories, with paths ./src ./dest:
  ${name} src/one/two/three.txt src/one/four.txt ./dest/imported
    ${pt_action}:  
      src/one/two/three.txt -> dest/imported/one/two/three.txt
      src/one/four.txt -> dest/imported/one/four.txt

  ${name} -x one src/one/two/three.txt src/one/four.txt ./dest/imported
    ${pt_action}:  
      src/one/two/three.txt -> dest/imported/two/three.txt
      src/one/four.txt -> dest/imported/four.txt

  ${name} -x one -x one/two src/one/two/three.txt ./dest/in
    ${pt_action}:  
      src/one/two/three.txt -> dest/in/three.txt
"""

DESC = '${action} file(s) (with history) from one hg repository to another.'


def exec_cmd(tvars, action):
    tvars['Action'] = tvars['action'].capitalize()

    parser = argparse.ArgumentParser(
                description=DESC.format(**tvars),
                formatter_class=argparse.RawDescriptionHelpFormatter,
                epilog=EPILOG.format(**tvars))
    parser.add_argument('files',
                        metavar='FILE',
                        type=str,
                        nargs='+',
                        help='file(s) to ${action}'.format(**tvars))
    parser.add_argument('dest_dir',
                        metavar='DEST_DIR',
                        help='dest directory')
    parser.add_argument('--strip-prefix',
                        '-x',
                        dest='strip_prefix',
                        action='append',
                        metavar='STRIP-PREFIX',
                        help='strip this prefix from source files')
    parser.add_argument('--pretend',
                        '-n',
                        action='store_true',
                        dest='pretend',
                        default=False,
                        help="Don't actually do anything, just show what would be done.")

    pargs = parser.parse_args()

    #print args
    try:
        success = cross_repo_copy(pargs.files,
                                  pargs.dest_dir,
                                  pargs.strip_prefix,
                                  pretend=pargs.pretend,
                                  action=action)
    except subprocess.CalledProcessError as e:
        print("An error occurred while executing the command: ")
        print('    ' + str(e.cmd))
        print('The return code was: %d' % e.returncode)
        print('The output of the command was:')
        print('    ' + ('    \n'.join([line
                                       for line in e.output.splitlines()])))
        success = False

    if success:
        sys.exit(0)
    sys.exit(-1)


def cmd_copy():
    tvars = {'action': 'copy',
             'name': 'hgcp',
             'pt_action': 'copies',
             'src_and': 'source and ',
            }
    exec_cmd(tvars, 'copy')


def cmd_move():
    tvars = {'action': 'move',
             'name': 'hgmv',
             'pt_action': 'moves',
             'src_and': '',
            }
    exec_cmd(tvars, 'move')



