#!/usr/bin/python

from optparse import OptionParser
from configobj import ConfigObj

import re
import paramiko
import ssh
import subprocess
import socket
import os.path
import copy

from console.console import Console
from connstring import ConnectionStringParser
from transfermanager import TransferManager

CONFIG_FILE_LOCATIONS = ["~/habitat.conf", "./habitat.conf", "../habitat.conf"]

def _build_parser():
    parser = OptionParser()
    parser.add_option("-d", action="store_true", dest="debug")
    parser.add_option("-u", action="store", dest="user")
    parser.add_option("-c", action="store", dest="connection_string")
    parser.add_option("-s", action="store", dest="server")
    return parser

def hidden_file_at(location):
    loc_parts = os.path.split(location)
    return loc_parts[0] + "/." + loc_parts[1]

def main():
    (opts, args) = _build_parser().parse_args()
    c = Console()
    tm = TransferManager()

    # Pull some general stuffs from the confi
    # file now.
    conf = None

    # Modify config file locations variable so that we
    # can also look for hidden files
    hidden_files = copy.deepcopy(CONFIG_FILE_LOCATIONS)
    for i in range(len(hidden_files)):
        hidden_files[i] = hidden_file_at(hidden_files[i])
    CONFIG_FILE_LOCATIONS.extend(hidden_files)

    for loc in CONFIG_FILE_LOCATIONS:
        loc = os.path.expanduser(loc)
        if os.path.exists(loc): 
            conf = ConfigObj(loc)
            break
    try:
        # If debug wasn't already set by the switch
        # option above, set it here
        if conf:
            p = conf["preferences"]
            if not tm.debug and p["debug"]=="true": 
                tm.debug = True
            tm.default_file_type = p["default_file_type"]
            tm.default_file_name = p["default_file_name"]

            # We will come to the use private/public key thing
            # a little bit later
            tm.public_private_key = p["use_public_private_key"]
    except KeyError:
        # Means that setting wasn't set.  Just continue since
        # the defaults will still be there anyways
        pass

    # Takes a string like username@address:port
    csp = ConnectionStringParser()
    possible_connection_string = None

    if opts.connection_string:
        possible_connection_string = opts.connection_string
    elif conf:
        server = None
        if opts.server:
            server = opts.server
        else:
            server = "default"

        try:
            possible_connection_string = conf["servers"][server]["connection_string"]
        except KeyError as e:
            print("Unknown server alias: %s, exiting..." % str(e))
            return
        try:
            tm.package_directory = conf["servers"][server]["package_directory"]
        except KeyError as e:
            print("Package directory did not exists for named server, exiting...")
            return
        try:
            tm.public_private_key = conf["servers"][server]["use_public_private_key"]
            if tm.public_private_key == "false":
                tm.public_private_key = False
            else:
                print("Using public private keys")
        except KeyError as e:
            # All this means is that they don't have keys set up,
            # we can safely ignore it
            pass

    else:
        # Worst case scenario, make the user enter it line by line
        addr = c.console("Address: ")
        port = c.console("Port[22]: ", default="22")

        if opts.user:
            user = opts.user
        else:
            user = c.console("Username: ")

        possible_connection_string = "%s@%s:%s" % (user, addr, port)

    # Now verify the connection string and exit
    # if it wasn't any good
    success, user, addr, port = csp.parse(possible_connection_string)
    if not success:
        print("Malformed connection string, exiting...")
        return

    # Get the user's password, making sure it's accurate
    if not tm.public_private_key:
        while True:
            passwd = c.console(prompt="Password: ", password=True)
            try:
                tm.connection = ssh.Connection(host=addr, username=user, password=passwd, port=port)
                break
            except paramiko.AuthenticationException:
                print("Invalid password")
            except socket.gaierror:
                print("Unable to connect to remote host, exiting...")
                return
    else:
        try:
            tm.connection = ssh.Connection(host=addr, username=user, port=port)
        except paramiko.AuthenticationException:
            print("Unable to authenticate with private key, exiting...")
            return

    if len(args) >= 1 and args[0] in ["push", "pull"]:
        # Just make sure name is assigned
        if len(args) > 1:
            name = args[1]
        else:
            name=tm.default_file_name

        if args[0] == "push":
            if conf: 
                print("Configuration file exists, searching for package: " + name)
                try:
                    files=conf["packages"][name]["files"].split(",")
                    files = [f.strip() for f in files if f.strip() != ""]
                except KeyError:
                    print("Unable to process package name.  Please edit your habitat.conf and try again, exiting...")
                    return

                # Make sure all files in the given package are
                # present
                print("Scanning files...")
                for f in files:
                    if not os.path.exists(f):
                        print("File %s not found, exiting..." % f)
                        return
                print("All files found.")
            else:
                # The name will be default here, just need
                # to find out if they have a conf or not.  If
                # they don't,this get's triggered, resulting
                # in a push all scenario to the default name package

                prompt = "\nWARNING:\nyou are about to push all files in the current directory to the default package!\n\nProceed[n]: "
                push_all = c.console(prompt, default="n")
                if push_all in "nN":
                    print("Aborting...")
                    return
                else:
                    files = ["."]

        if args[0] == "pull":
            print("Extracting %s's habbitat from %s:%s" % (user, addr, str(port)))
            tm.pull(name)
        else:
            # Must be push given the above check
            print("Pushing %s's current enviornment to %s:%s" % (user, addr, str(port)))
            tm.push(name, files)

        print("Done.")
    else:
        print("Please select either push or pull")

    tm.connection.close()

if __name__ == "__main__":
    try:
        main()
    except KeyboardInterrupt:
        print("\nKeyboard Interrupt detected, exiting...")
