import unittest
from habitat import *
from remote import Remote
from package import Package
from packagefactory import PackageFactory
import os.path
from remotefactory import RemoteFactory

class TestHabitat(unittest.TestCase):

    def setup(self):
        pass
    
    def test_nothing(self):
        pass

    def test_remote_init(self):
        r = Remote("adam@141.219.78.112:3333")
        self.assertEquals(r.user, "adam")
        self.assertEquals(r.port, 3333)
        self.assertEquals(r.addr, "141.219.78.112")

        # Make sure we are throwing an error
        # properly
        try:
            r = Remote()
        except ValueError as e:
            pass

        try:
            r = Remote("derp")
        except ValueError as e:
            pass
    
    def test_remote_connection(self):
        r = Remote("whitey@psg.mtu.edu:49931")
        c = r._get_connection("nintendex6993")
        if c:
            pass
        else:
            raise Exception

    def test_package(self):
        p = Package()
        self.assertEquals(p.full_path(), "./default.tar.gz")
        p = Package("hello", "/home/derp")
        self.assertEquals(p.full_path(), "/home/derp/hello.tar.gz")

    def test_package_factory(self):
        success, p = PackageFactory().make("derp", '.', ["wooww"])
        self.assertFalse(success)
        success, p = PackageFactory().make("derp", '.', [])
        self.assertTrue(success)
        self.assertTrue(p.compress)

    def test_full_series(self):
        success, p = PackageFactory().make("test", ".", ["__init__.py"])
        self.assertTrue(success)
        p.compress()
        self.assertTrue(os.path.exists("test.tar.gz"))
        r = Remote("whitey@psg.mtu.edu:49931")
        r.put(p, "nintendex6993")
        p.uncompress()
        p.remove()
        p.set_local()
        p.remove()

    def test_making_from_config_file(self):
        p = PackageFactory()
        success, r = p.make_from_config("~/.habitat.conf", "derp")
        self.assertFalse(success)

        success, r = p.make_from_config("~/.habitat.conf", "default")
        self.assertTrue(success)
        self.assertEquals(r._files(), "~/.vimrc ~/.bashrc ~/.habitat.conf")

        success, r = p.make_from_config("derp", "default")
        self.assertFalse(success)
        
    def test_remote_factory(self):
        p = RemoteFactory()
        success, r = p.make_from_config("~/.habitat.conf", "wopr")
        self.assertTrue(success)
        self.assertTrue(r.keys)

        success, r = p.make_from_config("~/.habitat.conf", "default")
        self.assertTrue(success)
        self.assertFalse(r.keys)

        success, r = p.make_from_config("~/.habitat.conf", "derp")
        self.assertFalse(success)

        success, r = p.make_from_config("derp")
        self.assertFalse(success)

    def test_pull(self):
        r = Remote("whitey@psg.mtu.edu:49931")
        success, p = PackageFactory().make(files=["__init__.py"])
        self.assertTrue(success)
        p.compress()
        r.put(p, "nintendex6993")
        subprocess.Popen("rm -f __init__.py", shell=True).wait()
        self.assertFalse(os.path.exists("__init__.py"))
        r.pull(p, "nintendex6993")
        self.assertTrue(os.path.exists("__init__.py"))

    def test_push(self):
        r = Remote("whitey@psg.mtu.edu:49931")
        success, p = PackageFactory().make(files=["__init__.py"])
        self.assertTrue(success)
        r.push(p, "nintendex6993")

if __name__ == "__main__":
    unittest.main()
