#!/usr/bin/python
import os.path
from optparse import OptionParser

from remotefactory import RemoteFactory
from packagefactory import PackageFactory
from remote import Remote
from console.console import Console

COMMANDS = ["push", "pull", "deploy"]

def _build_parser():
    parser = OptionParser()
    parser.add_option("-c", action="store", dest="connection_string")
    parser.add_option("-s", action="store", dest="server")
    parser.add_option("-k", action="store_true", dest="keys", default=False)
    parser.add_option("-d", action="store", dest="directory", default=".")
    parser.add_option("-f", action="store", dest="files", default="")
    return parser

def main():
    (opts, args) = _build_parser().parse_args()
    config_file = None
    c = Console(">>> ")

    if os.path.exists(os.path.expanduser("~/.habitat.conf")):
        config_file = "~/.habitat.conf"
    
    # Generate the remote server and package
    r = None
    if opts.connection_string:
        try:
            r = Remote(opts.connection_string, opts.keys)
        except ValueError:
            print("The connection string was invalid")
            return
    elif config_file:
        server = None
        if opts.server:
            server = opts.server
        else:
            server = "default"

        try:
            success, r = RemoteFactory().make_from_config(config_file, server)
            if not success:
                print("Unable to generate a remote server from the given name")
                return
        except ValueError:
            print("The connection string was invalid")
            return
    else:
        # Worst case scenario, make the user enter it line by line
        addr = c.console("Address: ")
        port = c.console("Port[22]: ", default="22")

        if opts.user:
            user = opts.user
        else:
            user = c.console("Username: ")
        
        # Don't know why this is allowed right now, quick fix
        if not user:
            print("Username required")
            return

        possible_connection_string = "%s@%s:%s" % (user, addr, port)
        keys = c.console("Do you have public/private keys set up? ")[0] in "yY"

        try:
            r = Remote(possible_connection_string, keys)
        except ValueError:
            print("The connection string was invalid")
            return

    
    # At this point, the remote should be defined.  If it isn't then exit
    #if not r:
    #   print("No server defined, exiting...")

    #print(r.user + " " + r.addr + " " + str(r.port) + " " + str(r.keys))

    if len(args) < 1 or args[0] not in COMMANDS:
        print("Invalid command or not enough arguments")
        return

    passwd = None
    if not r.keys:
        passwd = c.console("[%s]Password: " % r.addr, password=True)

    if len(args) < 3:
        # These are only for push and pull
        if len(args) < 2:
            if opts.files:
                success, p = PackageFactory().make("default", opts.directory, trim_files_list(opts.files), args[0] == "pull")
            else:
                success, p = PackageFactory().make_from_config(config_file, nocheck=args[0] == "pull")
        else:
            if opts.files:
                success, p = PackageFactory().make(args[1], opts.directory, trim_files_list(opts.files), args[0] == "pull")
            else:
                success, p = PackageFactory().make_from_config(config_file, args[1], nocheck=args[0] == "pull")

        if not success:
            print("Unable to make package")
            return

    if args[0] == "push":
        r.push(p, passwd)
    elif args[0] == "pull":
        r.pull(p, passwd)
    else:
        # Must be 3 arguments at least: deploy remote package
        if len(args) < 3:
            print("Please provide: deploy [remote_server] [package_name]")
        
        success, deploy = RemoteFactory().make_from_config(config_file, args[1])
        if not success:
            try:
                deploy = Remote(args[1], opts.keys)
            except ValueError:
                print("Improper connection string for server 2")
                return

        if not deploy:
            print("The second server could not be generated")
            return
        
        if opts.files:
            success, p = PackageFactory().make(args[2], opts.directory, trim_files_list(opts.files), True)
        else:
            success, p = PackageFactory().make_from_config(config_file, args[2], True)

        if not success:
            print("Package could not be created, exiting...")
            return 

        passwd_remote = None
        if not deploy.keys:
            passwd_remote = c.console("[%s]Password: " % deploy.addr, password = True)
        r.get(p, passwd)
        deploy.put(p, passwd_remote)
        p.uncompress()
        p.remove()
        p.set_local()
        p.remove()
    
def trim_files_list(files):
    """Convinience function for opts.files"""
    return [f.strip() for f in files.split(",")]

if __name__ == "__main__":
    main()
