from connstring import ConnectionStringParser
import ssh
import paramiko
import socket

class Remote(object):
    def __init__(self, connection=None, keys=False):
        if not connection:
            raise ValueError("A connection string is necessary") 
        self.csp = ConnectionStringParser()
        success, user, addr, port = self.csp.parse(connection)
        if not success:
            raise ValueError("The connection string was invalid")
        else:
            self.user = user
            self.addr = addr
            self.port = int(port)
        
        self.keys = keys
        
    def _get_connection(self, passwd):
        connection = None

        if not passwd and not self.keys:
            raise RuntimeError("A password is necessary for this operation") 
        elif self.keys:
            try:
                connection = ssh.Connection(host=self.addr,
                        username=self.user,
                        port=self.port)
            except paramiko.AuthenticationException:
                raise RuntimeError("Your keys were incorrect")
            except socket.gaierror:
                raise RuntimeError("Remote host does not exist or is down")
        else:
            try:
                connection = ssh.Connection(host=self.addr,
                        username=self.user,
                        password=passwd,
                        port=self.port)
            except paramiko.AuthenticationException:
                raise RuntimeError("Your password was incorrect")
            except socket.gaierror:
                raise RuntimeError("Remote host does not exist or is down")

        return connection

    def get(self, package, passwd=None):
        c = self._get_connection(passwd)

        try:
            c.get(package.full_path())
            package.set_local()
            return True
        except IOError as e:
            return False

    def put(self, package, passwd=None):
        c = self._get_connection(passwd)

        try:
            c.put(package.full_name(), package.full_path())
            package.set_remote(c)
            return True
        except IOError as e:
            print("Unable to push to remote server, possibly bad permissions")
            return False

    def execute(self, command, passwd):
        c = self._get_connection(passwd)
        c.execute(command)

    # Convinience function
    def push(self, p, passwd=None):
        p.compress()
        self.put(p, passwd)
        p.set_local()
        p.remove()
    
    def pull(self, p, passwd=None):
        self.get(p, passwd)
        p.uncompress()
        p.remove()
